/*********************************************************************
 *  SPDX-License-Identifier: MIT									 *
 *  Copyright (C) 2021-2023 by Helge Kortendieck  alan@kasmithee.de  *
 *                                                                   *
 *  This file is part of tlmb.                                       *
 *********************************************************************/

#include "rw.h"

	bool bRwErr;					// error while importing channel list

	int iRw;
	int iRwCB;						// return from create backup: -1=undefined, 0=OK, 1=not writable, 2=bak to bak2 failed, 3=file to bak failed, 4=bak2 to bak failed, 5=del bak2 failed
	int iRwCurLineLen;				// current line length of channel line currently processed, shrinks while we take values from begin to end...
	int iRwD;
	int iRwD2;
	int iRwFS;						// file status in File Save
	int iRwFieldCounter;			// counts # of processe field in channel line, must be 13 for a complete line
	int iRwIx;						// index

	QString sRwCurGrp;				// current group while importing channel list
	QString sRwD;
	QString sRwD2;
	QString sRwD3;
	QString sRwD4;
	QString sRwErrTit;				// error title
	QString sRwErrTxt;				// error text
	QString sRwFileName;			// full path and name of file to save

	QStringList slRwD;
	QStringList slRwD2;
//	QStringList slRwDomains;		// domain list read from file, 1 domain per line
	QStringList slRwErr;			// list with errors

	StrSong strRwSongD;				// song struct entity
	QList<StrSong> liRwSongStr;		// list with song structs

// *******************************************************************
// **  Dir functions										**
// *******************************************************************
int rwStatus(const QString& sRwD)		// check status: -1=does not exist, 1xx=dir/0xx=file, 1x=readable, 1=writable
{
	QFileInfo qfi(sRwD);
	if (qfi.exists() == false) {				// -1 = does not exist
		iRwD = -1;
		return iRwD;
	}

	iRwD = 0;
    if (qfi.isDir() == true) {					// 100 = is dir (only dir and file???)
        iRwD += 100;
    }

    if (qfi.isReadable() == true) {				// +10 = is readable
        iRwD += 10;
	}

    if (qfi.isWritable() == true) {				// +1 = is writable
        iRwD += 1;
	}

	return iRwD;
}

QStringList rwGetListDirs(const QString& sRwD)    	// get list of readable dirs excluding 'playlists*' and any 'ignore*' dir
{
    QDir qdir(sRwD);
        qdir.setFilter(QDir::Dirs | QDir::Readable | QDir::NoDotAndDotDot);
        qdir.setSorting(QDir::Name);
    slRwD = qdir.entryList();
    for (int i = 0; i < slRwD.size(); ++i) {
        if (slRwD.at(i).left(9) == "playlists" or slRwD.at(i).left(6) == "ignore") {
            slRwD.remove(i);
        }
    }
    return slRwD;
}

QStringList rwGetListPlaylists(const QString& sRwD) // get list of readable playlists in given directory
{
    QDir qdir(sRwD);
        qdir.setFilter(QDir::Files | QDir::Readable);
        slRwD.clear();
        slRwD << "*.m3u";
        qdir.setNameFilters(slRwD);
        qdir.setSorting(QDir::Name);
    return qdir.entryList();
}

QStringList rwGetListSongs(const QString& sRwD)  	// get list of readable songs in given directory
{
    QDir qdir(sRwD);
        qdir.setFilter(QDir::Files | QDir::Readable);
        slRwD.clear();
        slRwD <<  "*.mp3" << "*.ogg" << "*.flac" << "*.wav" << "*.wma" << "*.aac" << "*.ts" << "*.mkv" << "*.mp4" << "*.webm";
        qdir.setNameFilters(slRwD);
        qdir.setSorting(QDir::Name);
    return qdir.entryList();
}

// *******************************************************************
// **  File functions										**
// *******************************************************************
/*
int rwFileStatus(const QString& sRwD)		// check file status: 0=does not exist, 100=file, +10=readable, +1=writable
{
    QFileInfo qfi(sRwD);
    if (qfi.exists() == false) {				// 0 = does not exist
        iRwD = 0;
        return iRwD;
    }

    if (qfi.isFile() == true) {					// 1000 = is file
        iRwD = 1000;
    }

    if (qfi.isReadable() == true) {				// +100 = is readable
        iRwD += 100;
    }

    if (qfi.isWritable() == true) {				// +10 = is writable
        iRwD += 10;
    }

    return iRwD;
}
*/
/*
bool rwCommonFileDelete(const QString& sRwD3)		// delete file
{
	QFile f(sRwD3);
	return (f.remove());
}

bool rwCommonFileReName(const QString& sRwD3, const QString& sRwD4)	// rename file type: sRwD3=old, sRwD4=new
{
	QFile f(sRwD3);
	return (f.rename(sRwD4));
}
*/
/*
// *******************************************************************
// **  backup file functions										**
// *******************************************************************
int rwCommonBackupCreate(const QString& sRwD2)	// create a backup before writing the current file (replacing existing backup), -1=undefined, 0=OK, 11=not writable, 21=bak2 still exist=previous error, 31=bak to bak2 failed, 41=file to bak failed, 51=bak2 to bak failed, 61=del bak2 failed
{
	iRwCB = -1;									// return from create backup: -1=undefined, 0=OK, 11=not writable, 21=bak2 still exist=previous error, 31=bak to bak2 failed, 41=file to bak failed, 51=bak2 to bak failed, 61=del bak2 failed
	iRwBak = -1;								// -1=undefined, 0=could not be created (no original file), 1=exists, 2=just newly created, 3=try to create from *.bak2
	iRwBak2 = -1;								// -1=undefined, 0=could not be created (no *.bak file), 1=exists, 2=just newly created

	// *****  step 1)  only proceed, if we already have a writable domain file
	if (rwCommonFileStatus(sRwD2) > 0) {				// file exists - backup needed or 'is dir' = error
		if (rwCommonFileStatus(sRwFileName) < 10) {	// domain file not writable - abort
			return 11;
		}
	} else {									// we should do step 1) and 2) ( 3) )here, but it get's a bit messy
		bRwCont = true;
	}

	// *****  step 2)  if *.bak2 exists - abort, we failed previously, manual interaction required
	if (rwCommonFileStatus(sRwD2 + ".bak2") > 1) {	// *.bak2 still exists, we failed previously, manual interaction required, abort
		return 31;
	}

	// *****  step 3)  if *.bak exists, rename to *.bak2, ignore if it does not exists
	if (rwCommonFileStatus(sRwD2 + ".bak") > 1) {		// backup exists, try to re-name
		iRwBak = 1;
		if (rwCommonFileReName(sRwD2 + ".bak", sRwD2 + ".bak2") == true) {
			iRwBak2 = 2;
		} else {									// error step 2)
			return 41;
		}
	} else {									// no backup file (yet) continue
		iRwBak = 0;
		iRwBak2 = 0;
	}

	// *****  step 4) rename old file to *.bak, we already know, that we have a writable file
	if (rwCommonFileReName(sRwD2, sRwD2 + ".bak") == false) {	// creating backup failed
		if (iRwBak2 == 2) {					// try to rename *.bak2 to *.bak
			iRwBak = 3;
		} else {
			return 51;
		}
	} else {
		bRwCont = true;
	}

	// *****  step 5) rename *.bak2 back to *.bak, if we could not create *.bak from file
	if (iRwBak == 3) {
		if (rwCommonFileReName(sRwD2 + ".bak2", sRwD2 + ".bak") == false) {
			return 61;									// leave the *.bak2 file untouched for now - manual intervention needed
		} else {
			iRwBak = 2;									// successfully created *.bak from *.bak2 - *.bak2 can be deleted
		}
	} else {
		bRwCont = true;
	}

	// *****  step 6) del *.bak2 file
	iRwD = rwCommonFileStatus(sRwD2 + ".bak2");		// check for existing file
	if (iRwD > 0 and iRwBak != 3) {				// *.bak2 exists and can be deleted (*.bak was re-created)
		if (rwCommonFileDelete(sRwD2 + ".bak2") == false) {
			return 71;
		} else {
			bRwCont = true;
		}
	}

	return 0;
}

QString rwBackupErrorTxt(int iRwD, const QString& sRwD4)	// return a human readable error message from given error code, -110 from rwBackupCreate, -xxx from rwDomainFileSave
{
	switch (iRwD) {
		case 0:
			sRwErrTxt = QObject::tr("The error is, that this is _NOT_ an error!\nSorry, this must never happen!");
			break;
		case 11:
		    sRwErrTxt = QObject::tr("The existing file\n"
			                        "%1\n"
			                        "is not writable!\n"
			                        "E. g. for renaming or SaveAs.").arg(sRwD4);
			break;
		case 21:
		    sRwErrTxt = QObject::tr("A *.bak version of\n"
			                        "%1\n"
			                        "exists without file.\n"
			                        "This indicates a previous error.\n"
			                        "Manual interaction is now required!").arg(sRwD4);
			break;
		case 31:
		    sRwErrTxt = QObject::tr("A *.bak2 version (2nd backup) of\n"
			                        "%1\n"
			                        "still exists.\n"
			                        "This indicates a previous error.\n"
			                        "Manual interaction is required!").arg(sRwD4);
			break;
		case 41:
		    sRwErrTxt = QObject::tr("Temporary renaming the existing backup\n"
			                        "%1\n"
			                        "to *.bak2 failed!").arg(sRwD4 + ".bak");
			break;
		case 51:
		    sRwErrTxt = QObject::tr("Creating a backup from\n"
			                        "%1\n"
			                        "failed!").arg(sRwD4);
			break;
		case 61:
		    sRwErrTxt = QObject::tr("Recovering from creating a backup: rename\n"
			                        "%1\n"
			                        "to *.bak failed!\n"
			                        "Manual interaction is now required!").arg(sRwD4 + ".bak2");
			break;
		case 71:
		    sRwErrTxt = QObject::tr("Deleting the temporary 2nd backup\n"
			                        "%1\n"
			                        "failed!\n"
			                        "Manual interaction is now required!").arg(sRwD4 + ".bak2");
			break;
		case 81:
		    sRwErrTxt = QObject::tr("Re-creating\n"
			                        "%1\n"
			                        "from backup *.bak failed!\n"
			                        "Manual interaction is now required!").arg(sRwFileName);
			break;
		default:
		    sRwErrTxt = QObject::tr("Undefined error!\n"
			                        "Sorry, this must never happen!\n"
			                        "Aborting (hopefully)...");		// also covers -1 case
	}
	return sRwErrTxt;
}

// *******************************************************************
// **  domain list file - open										**
// *******************************************************************
QList<StrDomain> rwDomainFileOpen(const QString& sRwD)		// open file sRwD: read domain list file, setup list and return it
{																// SetupList (checks for errors during readFile)
	sRwFileName = sRwD;											// file name
	return rwDomainSetupList(rwDomainFileRead(sRwFileName));	// 1st read file, then separate domains and entries per domain
	                                                            // SetupList checks for read errors in FileRead and returns error in list
}

QStringList rwDomainFileRead(const QString& sRwD)			// read/import domain list from file, read line by line
{
	slRwDomains.clear();
	QFile qf(sRwD);
	if (!qf.open(QIODevice::ReadOnly | QIODevice::Text)) {	// if we cannot open the file for reading, abort
		slRwDomains.append("qublc");
		slRwDomains.append("Error opening file");
	} else {
		QTextStream in(&qf);						// initiate the stream to read the channel list line by line at first
		while (!in.atEnd()) {						// read the file (textstream)
			slRwDomains.append(in.readLine());		// read line by line
		}
		qf.close();									// close channel list file at EOF
	}
//  cout << "readFile - last line: " << slRwDomains.last().toStdString() << endl;                                               // *****  debug  *****
	return slRwDomains;
}

QList<StrDomain> rwDomainSetupList(const QStringList& slRwD)	// setup the domain list from the file read, separate entries
{
	slRwDomains = slRwD;
	liRwDomains.clear();
	slRwErr.clear();

	if (slRwDomains.count() == 0) {						// empty file !
		return liRwDomains;								// return empty list - error message in qublc.cpp
	}

	if (slRwDomains.at(0) == "qublc") {					// error opening file for reading, we set this in rwDomainFileRead
		strRwDomainD.note = "qublc: err";
		strRwDomainD.dltop = slRwDomains.at(1);
		strRwDomainD.err = true;
		liRwDomains.append(strRwDomainD);
		return liRwDomains;
	} else {											// success reading file - create struct domain list
		for (int i = 0; i < slRwDomains.count(); ++i) {		// iterate over just loaded domain file list and extract domain data sets - 1 line = 1 domain entry set
			slRwDomainFields.clear();
			sRwD = slRwDomains.at(i);
			if (sRwD.length() > 0 and sRwD.left(1) != "#") {	// skip empty lines and lines starting with # (=comments)
				iRwD = sRwD.count("|");
				if (iRwD == 6) {								// looks like a valid entry...
					for (int i2 = 0; i2 < 6; ++i2) {
						iRwD2 = sRwD.indexOf("|");
						if (iRwD2 == 0) {							// '|' is first = empty field
							slRwDomainFields.append("");
							sRwD.remove(0,1);
						} else {
							slRwDomainFields.append(sRwD.left(iRwD2));
							sRwD.remove(0,iRwD2 + 1);
						}
					}
					if (sRwD.length() > 0) {
						slRwDomainFields.append(sRwD);
					} else {
						slRwDomainFields.append("");				// last field empty
					}
				} else {										// invalid entry
				}

				if (slRwDomainFields.count() != 7) {					// we store 7 values per domain - could not retrieve 7 back (this is for qublc file format V01, can theoretically change in the future)
					strRwDomainD.index = i;
					strRwDomainD.dltop = "";
					strRwDomainD.dl2 = "";
					strRwDomainD.dl3 = "";
					strRwDomainD.include = "";
					strRwDomainD.lastchkd = "";
					strRwDomainD.note = "qublc: err";
					strRwDomainD.recfrom = "";
//					strRwDomainD.sort = "";
					strRwDomainD.err = true;
					liRwDomains.append(strRwDomainD);
				} else {											// all domain fields there - define entry
					strRwDomainD.index = i;
					strRwDomainD.dltop = slRwDomainFields.at(0);
					strRwDomainD.dl2 = slRwDomainFields.at(1);
					strRwDomainD.dl3 = slRwDomainFields.at(2);
					strRwDomainD.include = slRwDomainFields.at(3);
					strRwDomainD.lastchkd = slRwDomainFields.at(4);
					strRwDomainD.note = slRwDomainFields.at(5);
					strRwDomainD.recfrom = slRwDomainFields.at(6);
//					strRwDomainD.sort = "";
					strRwDomainD.err = false;

//cout << "setupDomainList: " << QString::number(i).toStdString() << strRwDomainD.dl3.toStdString() << endl;                                        // *****  debug
                    liRwDomains.append(strRwDomainD);
				}
			} else {
			}
		}
	}
//cout << "setupDomainList: " << QString::number(liRwDomians.count()).toStdString() << endl;                                        // *****  debug
	return liRwDomains;
}

// *******************************************************************
// **  domain list file - save										**
// *******************************************************************
QStringList rwDomainFileSave(const QString& sRwD, const QList<StrDomain>& liRwDomainsD)	// prepare save domain list: path+name, domain list struct
{
//cout << "rwDomainFileSave - reached" << QString::number(liRwDomainsD.count()).toStdString() << endl;                                               // *****  debug  *****
	sRwFileName = sRwD;								// file name
	liRwDomains = liRwDomainsD;						// list of domains with details
	slRwD.clear();
	iRwFS = rwCommonFileStatus(sRwFileName);
	if (iRwFS > 0) {								// target exists already - action required
		if (iRwFS == 1) {								// is dir, not file > abort
			slRwD.append("99");
			slRwD.append(QObject::tr("Error given file path+name is a directory, not file!"));
			return slRwD;
		} else {										// file exists
			if (iRwFS > 9) {								// writeable file, try to create backup
				iRwD2 = rwCommonBackupCreate(sRwFileName);
				if (iRwD2 != 0) {								// we had errors while creating the backup > abort
					slRwD.append("99");
					slRwD.append(rwBackupErrorTxt(iRwD2, sRwFileName));
					return slRwD;
				}
			} else {										// file not writable - abort
				slRwD.append("99");
				slRwD.append(QObject::tr("Error file exists and is not writeable.\nAborting!"));
				return slRwD;
			}
		}
	}

	// write Domain file
	if (rwDomainFileWrite(sRwFileName, liRwDomains) == true) {	// success
		slRwD.append("0");
		slRwD.append(QObject::tr("Domain list written as:\n%1").arg(sRwFileName));
	} else {										// error writing domain list to file
		slRwD.append("99");
		sRwD2 = QObject::tr("Writing the domain list to file\n%1\nfailed!").arg(sRwFileName);
		if (iRwFS > 0) {													// we created a backup! try to re-create old file from *.bak
			if (rwCommonFileReName(sRwFileName + ".bak", sRwFileName) == false) {	// *.bak to file failed
				sRwD2 = sRwD2 + "\n" + rwBackupErrorTxt(81, sRwFileName);	// error re-creating old domain list from *.bak
		    } else {
			    sRwD2 = sRwD2 + "\n" + QObject::tr("Successfully re-created the old file\n%1\nfrom backup.").arg(sRwFileName);	// error re-creating old domain list from *.bak
		    }
		}
		slRwD.append(sRwD2);
	}
	return slRwD;
}

bool rwDomainFileWrite(const QString& sRwD, const QList<StrDomain>& liRwDomainsD)	// write domain list file, full path+name & domain list
{
//cout << "rwDomainFileWrite - reached" << QString::number(liRwDomainsD.count()).toStdString() << endl;                                               // *****  debug  *****
	QFile file(sRwD);
	if (file.open(QIODevice::WriteOnly)) {
		QTextStream stream(&file);

		stream << "# This is a domain list file from qublc," << "\n";						// create file header
		stream << "# a small program to manage domains to be blocked by unbound." << "\n";
		stream << "# qublc can create a *.conf file for unbound from this list." << "\n";
		stream << "# File format V01." << "\n";
		stream << "# Please do not edit manually. qublc's import function is not overly robust! :-)" << "\n";
		stream << "#" << "\n";

		for (int i = 0; i < liRwDomainsD.count(); ++i) {		// do not store 'index' and 'sort' ('dltop.dl2.dl3') - re-create on the fly 'index' on import, 'sort' during each sort
			strRwDomainD = liRwDomainsD.at(i);
			stream << strRwDomainD.dltop << "|" << strRwDomainD.dl2 << "|" << strRwDomainD.dl3 << "|" << strRwDomainD.include << "|" << strRwDomainD.lastchkd << "|" << strRwDomainD.note << "|" << strRwDomainD.recfrom << "\n"; // add entry
		}
		file.close();
		return true;
	} else {
		return false;
	}
}

// *******************************************************************
// **  unbound conf file - save										**
// *******************************************************************
QStringList rwUnboundFileSave(const QString& sRwD, const QList<StrDomain>& liRwDomainsD)	// create unbound conf file: path+name, domain list struct
{
	sRwFileName = sRwD;								// file name
	liRwDomains = liRwDomainsD;						// list of domains with details
	slRwD.clear();
//cout << "rwUnboundFileSave - reached: " << QString::number(liRwDomains.count()).toStdString() << endl;                                               // *****  debug  *****

	// target is file and writable?
	iRwFS = rwCommonFileStatus(sRwFileName);
	if (iRwFS > 0) {								// entry exists (file or dir), need to create backup
		if (iRwFS == 1) {								// is dir, not file > abort
			slRwD.append("99");
			slRwD.append(QObject::tr("Error given file path+name is a directory, not file!"));
			return slRwD;
		} else {										// file exists, try to create backup
			iRwD2 = rwCommonBackupCreate(sRwFileName);
			if (iRwD2 != 0) {							// we had errors > abort
				slRwD.append("99");
				slRwD.append(rwBackupErrorTxt(iRwD2, sRwFileName));
				return slRwD;
			}
		}
	}

	// write Domain file
	if (rwUnboundFileWrite(sRwFileName, liRwDomains) == true) {	// success
		slRwD.append("0");
		slRwD.append(QObject::tr("Domain list written as:\n%1").arg(sRwFileName));
	} else {										// error writing domain list to file
		slRwD.append("99");
		sRwD2 = QObject::tr("Writing the domain list to file\n%1\nfailed!").arg(sRwFileName);
		if (iRwFS > 0) {													// we created a backup! try to re-create old file from *.bak
			if (rwCommonFileReName(sRwFileName + ".bak", sRwFileName) == false) {	// *.bak to file failed
				sRwD2 = sRwD2 + "\n" + rwBackupErrorTxt(81, sRwFileName);	// error re-creating old domain list from *.bak
			} else {
				sRwD2 = sRwD2 + "\n" + QObject::tr("Successfully re-created the old file\n%1\nfrom backup.").arg(sRwFileName);	// error re-creating old domain list from *.bak
			}
		}
		slRwD.append(sRwD2);
	}
	return slRwD;
}

bool rwUnboundFileWrite(const QString& sRwD, const QList<StrDomain>& liRwDomainsD)	// write unbound blacklist conf file, full path+name & domain list
{
	QFile file(sRwD);
	if (file.open(QIODevice::WriteOnly)) {
		QTextStream stream(&file);

		stream << "# DNS spoofing/blocking list for the Unbound DNS server." << "\n";						// create file header
		stream << "# Written with qublc - Q_t based U_nbound B_lackL_ist.C_onf." << "\n";
		stream << "#" << "\n";

		for (int i = 0; i < liRwDomainsD.count(); ++i) {
			strRwDomainD = liRwDomainsD.at(i);
			if (strRwDomainD.include == "x") {
				sRwD2 = "";
				if (strRwDomainD.dltop.length() > 0) {
					sRwD2 += strRwDomainD.dltop;

					if (strRwDomainD.dl2.length() > 0) {
						sRwD2.prepend(strRwDomainD.dl2 + ".");

						if (strRwDomainD.dl3.length() > 0) {
							sRwD2.prepend(strRwDomainD.dl3 + ".");
						} else {
						}
					} else {
					}
//cout << "rwUnboundFileWrite - reached: " << sRwD2.toStdString() << endl;                                               // *****  debug  *****
					stream << "local-zone: \"" << sRwD2 << "\" redirect" << "\n";
					stream << "local-data: \"" << sRwD2 << " A 0.0.0.0\"" << "\n";
//  for dnsmasq - not working? (also text at start of file needs to be removed)
//					stream << "address=/" << sRwD2 << "/0.0.0.0\n";
				} else {
				}
			}
		}
		file.close();
		return true;
	} else {
		return false;
	}
}
*/
