/*********************************************************************
 *  SPDX-License-Identifier: MIT									 *
 *  Copyright (C) 2021-2025 by Helge Kortendieck  alan@kasmithee.de  *
 *                                                                   *
 *  This file is part of tlmb.                                       *
 *********************************************************************/

#include "tlmb.h"

Tlmb::Tlmb(QMainWindow *parent) : 
		QMainWindow(parent)
{
    uiTlmb.setupUi(this);

	bMdPlayer = false;							// qt media player to retrieve songs meta data from
	bPlayer = false;
	bQuit = false;
	bSetDlgCreated = false;
	bStopPlaying = false;
	iWShown = 0;
	iWToShow = 0;

	playerCreate();								// create player qprocess, once at program start

	createActions();
	createMenu();
	createMenuCbPlAddTo();
	createMenuCbPlPlaying();
	createMenuTwSongs();
//	createToolBar();
//	createStatusBar();

	timersCreate();								// create timers

	this->setWindowIcon(QIcon("/usr/share/icons/hicolor/scalable/apps/tlmb.svg"));

	QScreen *screen = QGuiApplication::primaryScreen();
	rFrame = screen->geometry();
	iScrX = rFrame.width();						// screen size horizontal (pixels)
	iScrY = rFrame.height();					// screen size vertical (pixels)

    settingsRead();
}

Tlmb::~Tlmb()
{
}

//  *********************************************
//  **  settings                               **
//  *********************************************
void Tlmb::settingsRead()
{
	QSettings settings("tlmb", "tlmb");
	
	settings.beginGroup("main");
		QPoint pos = settings.value("pos", QPoint(200, 200)).toPoint();
		QSize size = settings.value("size", QSize(400, 400)).toSize();

		sLastMusicType = settings.value("playingMusicTypeLast", "").toString(); 	    // music type selected last session, name
		sPlayMusicType = sLastMusicType;													// needed for unique population routine (at start and dir change)
		sLastArtist = settings.value("playingArtistLast", "").toString();      			// artist selected last session, name
		sPlayArtist = sLastArtist;
		sLastPlRoot = settings.value("playingPlRootLast", "").toString();      			// root playlist selected last session, name
		sPlayPlRoot = sLastPlRoot;
		sLastPlMusicType = settings.value("playingPlMusicTypeLast", "").toString(); 	// music type playlist selected last session, name
		sPlayPlMusicType = sLastPlMusicType;
		sLastPlArtist = settings.value("playingPlArtistLast", "").toString();      			// artist playlist selected last session, name
		sPlayPlArtist = sLastPlArtist;
		
		iLastCbPlay = settings.value("playingCbLast", 1).toInt(); 							// combobox music was last played from: 1=Artist/Band, 2=Playlists root dir, 3=Playlists music typ, 4=Playlists Artist/band
//		sPlayPl = settings.value("playingPlLast", "").toString();						// playlist selected last session, name - "" when playing from artist dir

		iLastCbPlAddTo = settings.value("addToPlCbLast", 0).toInt(); 						// last active add-to playlist comboBox: 5=Playlists root dir, 6=Playlists music typ, 7=Playlists Artist/band
//		iAddToPlCb = iLastAddToPlCb;															// required for population (dirRootChgd)
//		if (iAddToPlCb > 4 and iAddToPlCb < 8) {
		sAddToPl = settings.value("addToPlLast", "").toString();						// last set add-to playlist, name
//			sAddToPl = sLastAddToPl;															// required for population (dirRootChgd)
//		} else {
//			sAddToPl = "";
//		}

		iColWidth0 = settings.value("twSongsWidthCol0", 50).toInt(); 						// songs table widget: width column 0, name
		iColWidth1 = settings.value("twSongsWidthCol1", 50).toInt(); 						// songs table widget: width column 1, typ
	settings.endGroup();

    settings.beginGroup("settingsDlg");
        sRootDir = settings.value("rootdir", "").toString();  		  						// path to music collection's root directory
        if (settings.value("startFullScreen", true).toBool() == true ) {  					// how to start: fullscreen, maximized, last size
			this->showFullScreen();
		} else if (settings.value("startMaximized", false).toBool() == true ) {
			this->showMaximized();
		} else if (settings.value("startLastSize", false).toBool() == true ) {
			resize(size);
			move(pos);
			this->show();
		}
//		uis.fcb_Font = settings.value("fontname", "").toString();							// font name only for the combo boxes and the song list
        iFontSize = settings.value("fontSize", 12).toInt();       						    // font size only for the combo boxes and the song list
        iIconSize = settings.value("iconSize", 24).toInt();       						    // size of indicator icons, e. g. to which combobox the songs belong to
		iPlayMode = settings.value("playMode", 1).toInt();       						    // play mode: 1=single song, 2=to end of list, 3= loop through list

		iPlayer = settings.value("player", 1).toInt();       						    	// player: 1=player 1, 2=player 2, 3=player 3
		switch (iPlayer) {
			case 1:
			    sPlayer = settings.value("player1", "").toString();
			    sPlayerOptions = settings.value("player1Options", "").toString();
			    break;
			case 2:
   			    sPlayer = settings.value("player2", "").toString();
			    sPlayerOptions = settings.value("player2Options", "").toString();
			    break;
			case 3:
   			    sPlayer = settings.value("player3", "").toString();
			    sPlayerOptions = settings.value("player3Options", "").toString();
			    break;
		}
		sPlayerOptions = sPlayerOptions.simplified();
		slPlayerOptions = sPlayerOptions.split(" ",Qt::SkipEmptyParts);
		if (sPlayer == "") {
			mbTxt_PlayerNoneDefined();
		} else {
			emit sigTlmbPlayerChkAvail(sPlayer, slPlayerOptions);				// check if sPlayer can be found - continue in playerAvail() with init() if found
		}
    settings.endGroup();
}

void Tlmb::settingsWrite()
{
	QSettings settings("tlmb", "tlmb");

	settings.beginGroup("main");
		settings.setValue("pos", pos());
		settings.setValue("size", size());

		settings.setValue("playingMusicTypeLast", uiTlmb.tlmb_cb_musicType->currentText());
		settings.setValue("playingArtistLast", uiTlmb.tlmb_cb_artist->currentText());
		settings.setValue("playingPlRootLast", uiTlmb.tlmb_cb_root_playlists->currentText());
		settings.setValue("playingPlMusicTypeLast", uiTlmb.tlmb_cb_musicType_playlists->currentText());
		settings.setValue("playingPlArtistLast", uiTlmb.tlmb_cb_artist_playlists->currentText());
		settings.setValue("playingCbLast", iPlayCb); 									// combobox music is played from: 1=Artist/Band, 2=Playlists root dir, 3=Playlists music typ, 4=Playlists Artist/band

/*		switch (iPlayCb) {										// set combobox last played from
			case 2:
				settings.setValue("playingPlLast", uiTlmb.tlmb_cb_root_playlists->currentText());
				break;
			case 3:
				settings.setValue("playingPlLast", uiTlmb.tlmb_cb_musicType_playlists->currentText());
				break;
			case 4:
				settings.setValue("playingPlLast", uiTlmb.tlmb_cb_artist_playlists->currentText());
				break;
			default:
				settings.setValue("playingPlLast", "");
		}
*/
		settings.setValue("addToPlCbLast", iAddToPlCb);
		settings.setValue("addToPlLast", sAddToPl);					// current playlist, name

		settings.setValue("twSongsWidthCol0", uiTlmb.tlmb_tw_Songs->columnWidth(0));		// songs table widget: width column 0, name
		settings.setValue("twSongsWidthCol1", uiTlmb.tlmb_tw_Songs->columnWidth(1));		// songs table widget: width column 1, typ
	settings.endGroup();

	settings.sync();
}

//  *********************************************
//  **  actions, menu, toolbar, statusbar      **
//  *********************************************
void Tlmb::createActions()
{
    actAbout = new QAction(QIcon("/usr/share/tlmb/icons/scalable/about.svg"), tr("&About tlmb"), this);
	actAbout->setShortcut(tr("Ctrl+A"));
	actAbout->setStatusTip(tr("Show the application's About dialog"));
	actAbout->setToolTip(tr("Show the application's about information"));
	connect(actAbout, SIGNAL(triggered()), this, SLOT(wAhShowAbout()));

	actExit = new QAction(QIcon("/usr/share/tlmb/icons/scalable/quit.svg"), tr("&Quit the program"), this);
	actExit->setShortcut(tr("Ctrl+Q"));
	actExit->setStatusTip(tr("Quit the application"));
	actExit->setToolTip(tr("Quit the application"));
	connect(actExit, SIGNAL(triggered()), this, SLOT(quitPrep()));

	actHelp = new QAction(QIcon("/usr/share/tlmb/icons/scalable/help.svg"), tr("&Help"), this);
	actHelp->setShortcut(tr("Ctrl+H"));
	actHelp->setStatusTip(tr("Show the application's online help"));
	actHelp->setToolTip(tr("Show the application's online help"));
	connect(actHelp, SIGNAL(triggered()), this, SLOT(wAhShowHelp()));

	actPlAddSong = new QAction(QIcon("/usr/share/tlmb/icons/scalable/plus-green.svg"), tr("A&dd this song to add-to playlist"), this);
	actPlAddSong->setStatusTip(tr("Add this song to current playlist"));
	connect(actPlAddSong, SIGNAL(triggered()), this, SLOT(plAddToAddSong()));

	actPlEditDeleteW = new QAction(QIcon("/usr/share/tlmb/icons/scalable/edit.svg"), tr("Close edit this pl&aylist dialog"), this);
	actPlEditDeleteW->setStatusTip(tr("Close edit current playlist editing window"));
	connect(actPlEditDeleteW, SIGNAL(triggered()), this, SLOT(wPlEditDelete()));

	actPlEditShowW = new QAction(QIcon("/usr/share/tlmb/icons/scalable/edit.svg"), tr("&Edit this playlist"), this);
	actPlEditShowW->setStatusTip(tr("Show current playlist to add songs to for editing"));
	connect(actPlEditShowW, SIGNAL(triggered()), this, SLOT(wPlEditShow()));

	actPlSetDeleteW = new QAction(QIcon("/usr/share/tlmb/icons/scalable/set.svg"), tr("Close the set playlist dialog"), this);
	actPlSetDeleteW->setStatusTip(tr("Close the set current playlist window"));
	connect(actPlSetDeleteW, SIGNAL(triggered()), this, SLOT(wAddToPlSetDelete()));

	actPlSetShowW = new QAction(QIcon("/usr/share/tlmb/icons/scalable/set.svg"), tr("Create new, rename or save-as playl&ist to add songs to"), this);
	actPlSetShowW->setStatusTip(tr("Set current playlist to add songs to"));
	connect(actPlSetShowW, SIGNAL(triggered()), this, SLOT(wAddToPlSetShow()));

	actSettings = new QAction(QIcon("/usr/share/tlmb/icons/scalable/settings.svg"), tr("&Settings"), this);
	actSettings->setShortcut(tr("Ctrl+S"));
	actSettings->setStatusTip(tr("Open the settings dialog"));
	actSettings->setToolTip(tr("Open the settings dialog"));
	connect(actSettings, SIGNAL(triggered()), this, SLOT(setDlgShow()));

/*
	actSongInfo = new QAction(QIcon("/usr/share/tlmb/icons/scalable/info.svg"), tr("File/Song I&nfo"), this);
	actSongInfo->setStatusTip(tr("Show file/song info"));
	connect(actSongInfo, SIGNAL(triggered()), this, SLOT(songInfoShow()));
*/
}

void Tlmb::createMenu()
{
	menuFile = menuBar()->addMenu(tr("&File"));
		menuFile->addAction(actExit);

	menuBar()->addSeparator();

	menuFile = menuBar()->addMenu(tr("&Settings"));
		menuFile->addAction(actSettings);

	menuBar()->addSeparator();
	
	menuHelp = menuBar()->addMenu(tr("&Help"));
		menuHelp->addAction(actHelp);
		menuHelp->addAction(actAbout);
}

void Tlmb::createMenuCbPlAddTo()					// right-click menu for playlist add to comboboxes
{
	menuCbPlAddTo = new QMenu;
	menuCbPlAddTo->clear();
	if (iWShown != 5) {									// set playlist widget
		menuCbPlAddTo->addAction(actPlSetShowW);
	} else {
		menuCbPlAddTo->addAction(actPlSetDeleteW);
	}
	if (iWShown != 6) {									// edit playlist widget
		menuCbPlAddTo->addAction(actPlEditShowW);
	} else {
		menuCbPlAddTo->addAction(actPlEditDeleteW);
	}
}

void Tlmb::createMenuCbPlPlaying()					// right-click menu for playlist playing comboboxes
{
	menuCbPlPlaying = new QMenu;
	menuCbPlPlaying->clear();
	if (iWShown != 4) {									// edit playlist widget
		menuCbPlPlaying->addAction(actPlEditShowW);
	} else {
		menuCbPlPlaying->addAction(actPlEditDeleteW);
	}
}

void Tlmb::createMenuTwSongs()						// songs table widget right-click menu
{
	menuTwSongsRightClick = new QMenu;
	menuTwSongsRightClick->clear();
/*	if (iWShown != 3) {									// set playlist widget
		menuTwSongsRightClick->addAction(actPlSetShowW);
	} else {
		menuTwSongsRightClick->addAction(actPlSetDeleteW);
	}
	if (iWShown != 4) {									// edit playlist widget
		menuTwSongsRightClick->addAction(actPlEditShowW);
	} else {
		menuTwSongsRightClick->addAction(actPlEditDeleteW);
	}*/
	menuTwSongsRightClick->addAction(actPlAddSong);
}

void Tlmb::createToolBar()
{
	toolBar = addToolBar(tr("Show toolbar"));
	toolBar->addAction(actExit);
//	toolBar->addSeparator();
}
/*
void Tlmb::createStatusBar()
{
	statusBar()->showMessage(tr("Ready"));
}

void Tlmb::updStatusBar(const QString& sD)
{
	statusBar()->showMessage(sD);
}
*/

//  **********  quit  **********
void Tlmb::quitPrep()                       		// quit clicked - clean up
{
/*	getCursorPos();
	QMessageBox msgBox;
	msgBox.setWindowTitle(tr("SaveQuit ?"));
	msgBox.setIcon(QMessageBox::Question);
	msgBox.setText(tr("\nSure to quit ?\n"));
	msgBox.addButton(tr("Yes, &Quit"), QMessageBox::AcceptRole);
	msgBox.addButton(tr("No, &Cancel"), QMessageBox::RejectRole);
	msgBox.move(iX, iY);
	if(msgBox.exec() == 0) {*/
		bQuit = true;
		settingsWrite();
		quitDo();
//	}
}

void Tlmb::quitDo()                         		// finally quit the program
{
	exit(0);
}

// **********************************************
// **  message box and it's messages           **
// **********************************************
int Tlmb::mbShow(int iD, const QString& sD, const QString& sD2, const QString& sD3, const QString& sD4, const QString& sD5)
{						// show QMessageBox at cursor position: type, title, text, button accept, button reject, button destruct
	QMessageBox msgBox;
	msgBox.setWindowTitle(sD);

	if (iD == 1) {
		msgBox.setIcon(QMessageBox::Information);
	} else if (iD == 2) {
		msgBox.setIcon(QMessageBox::Warning);
	} else if (iD == 3) {
		msgBox.setIcon(QMessageBox::Critical);
	} else if (iD == 4) {
		msgBox.setIcon(QMessageBox::Question);	// 2 buttons
	} else if (iD == 5) {
		msgBox.setIcon(QMessageBox::Question);	// 3 buttons, 0=yes, 1=cancel, 2=no
	} else {
		return 9;
	}

	msgBox.setText(sD2);

	msgBox.addButton(sD3, QMessageBox::AcceptRole);
	if (iD == 5) {
		msgBox.addButton(sD5, QMessageBox::DestructiveRole);
	}
	if (iD > 3) {
		msgBox.addButton(sD4, QMessageBox::RejectRole);
	}

	iX = iScrX/2 - msgBox.frameGeometry().width()/2;		// this adjustment should center the mbShow - but it doesn't !?
	if (iX < 0) {
		iX = 0;
	}
	iY = iScrY/2 - msgBox.frameGeometry().height()/2;
	if (iY < 0) {
		iY = 0;
	}
	msgBox.move(iX, iY);

	iRet = msgBox.exec();
	return iRet;
}

void Tlmb::mbTxt_AddToPlAddSongExists()				// show message box - songs already exists in add-to playlist
{
	mbShow(1, tr("Information - entry exists!"),
	          tr("The selected song already exists in the add-to playlist.\n"
	             "Nothing to do - item not added."), tr("&OK"), "", "");
}

void Tlmb::mbTxt_AddToPlAddSongFailed()				// show message box - failed to write updated add-to playlist
{
	mbShow(1, tr("Information - song not added!"),
	          tr("Failed to write the changed add-to playlist!?\n"
	             "Adding the selected song to the add-to playlist failed! Reason unknown."), tr("&OK"), "", "");
}

void Tlmb::mbTxt_AddToPlAddSongSuccess()			// show message box - song added to add-to playlist
{
	mbShow(1, tr("Information - song added."),
	          tr("The selected song\n"
				 "%1\n"
				 "was added to the add-to playlist.").arg(strSongD.sName), tr("&OK"), "", "");
}

void Tlmb::mbTxt_AddToPlNotFound()					// show message box - the (last) add-to playlist could not be found
{
	mbShow(1, tr("Information - add-to playlist not found."),
	          tr("Could not find the (last) used add-to playlist.\n"
				 "Please select another or create a new one,\n"
				 "if you want to add songs to a favorite playlist.\n"
				 "Internal: iAddToPlCb = %1").arg(iAddToPlCb), tr("&OK"), "", "");
}

void Tlmb::mbTxt_ArtistNoneFound()					// show message box - no artist found
{
	mbShow(1, tr("Information - no artist/band directory found."),
	          tr("Could not find any artist/band directory in the current music type directory.\n"
				 "You may want to read about the required music structure for tlmb in the online help.\n"
				 "Or just select another music type..."), tr("&OK"), "", "");
}

void Tlmb::mbTxt_MusicTypeNoneFound()				// show message box - no music type found
{
	mbShow(1, tr("Information - no music type directory found."),
	          tr("Could not find a valid music type directory in the current root directory.\n"
				 "You may want to read about the required music structure for tlmb in the online help (Ctrl+h).\n"
				 "The root directory can be set in the settings menu (Ctrl+s)."), tr("&OK"), "", "");
}

void Tlmb::mbTxt_PathRootInvalid()					// show message box - root directory invalid
{
	mbShow(1, tr("Information - root directory invalid !"),
	          tr("Attention, root directory is invalid.\n"
				 "It does not exist, is not a directory or is not readable.\n \n"
				 "Set the root directory via the settings dialog (Ctrl+s)\n"
				 "or read the online help (Ctrl+h) first."), tr("&OK"), "", "");
}

void Tlmb::mbTxt_PlLoadError(const QString& sD3, const QString& sD)	// show message box - could not read playlist - sD3 = play or add-to , sD = playlist full path & name
{
	mbShow(1, tr("tlmb error - could not read playlist!"),
	          tr("The %1 playlist:\n"
				 "%2\n"
				 "could not be read.").arg(sD3).arg(sD), tr("&OK"), "", "");
}

void Tlmb::mbTxt_PlayerNoneDefined()				// show message box - no player defined
{
	mbShow(1, tr("Information - no player defined !"),
	          tr("Please define the player to use under Settings."), tr("&OK"), "", "");
}

void Tlmb::mbTxt_PlayerNoneFound()					// show message box - defined player not found
{
	mbShow(1, tr("Information - defined player not found !"),
	          tr("Attention, could not find the player defined under Settings.\n"
				 "Without an external media player, no song can be played."), tr("&OK"), "", "");
}

void Tlmb::mbTxt_WPlEditEmpty()						// show message box - empty playlist - no need to open edit window
{
	mbShow(1, tr("tlmb - empty playlist!"),
	          tr("The internal playlist to add songs to is empty.\n"
				 "No need to open the edit playlist window."), tr("&OK"), "", "");
}

void Tlmb::mbTxt_WPlEditFailed()					// show message box - error - failed to save changes to playlist
{
	mbShow(1, tr("tlmb error - writing playlist failed!"),
	          tr("Failed to save changes to the playlist.\n"
				 "They are probably lost."), tr("&OK"), "", "");
}

void Tlmb::mbTxt_WPlEditSuccess()					// show message box - success - changes to playlist saved
{
	mbShow(1, tr("tlmb - playlist saved."),
	          tr("The changes to the playlist were successfully saved."), tr("&OK"), "", "");
}

// **********************************************
// **  init - bring things up...               **
// **********************************************
void Tlmb::init()
{
	cbIndicatorCreate();
	cbIndicatorPlayingClear();
	cbIndicatorPlAddToClear();

	twSongsHeaderSet();
	twSongsCon(true);
	customMenusSetPolicy();
	customMenusCon(true);
	
	if (dirRootChk() == true) {							// old root dir still valid - populate comboboxes and try to re-create last session
		uiTlmb.tlmb_l_rootDir->setText(sRootDir);
		cbPop();											// populate comboboxes
		sessionLastReCreate();								// re-create last session - if possible
	} else {											// last session's root dir no longer valid
		mbTxt_PathRootInvalid();							// show message
	}

	wMainFocusSet();
}

// ******************************************
// **  playlist add-to combobox functions  **
// ******************************************
void Tlmb::cbAddToPlArtistChgd(int iD)				// current index of add-to artist combobox has changed
{
	iAddToPlCbCurIdx = iD;
	iAddToPlCb = 7;
	sAddToPl = uiTlmb.tlmb_cb_plAddTo_artist->currentText();
	plAddToSetNew(sAddToPl);
}

void Tlmb::cbAddToPlArtistCon(bool bD)				// connect (true) / disconnect (false) add-to artist/band combobox signals
{
	if (bD == true) {
		connect(uiTlmb.tlmb_cb_plAddTo_artist, SIGNAL(currentIndexChanged(int)), this, SLOT(cbAddToPlArtistChgd(int)));	// set add-to playlist to this selection and iAddToPlCb to 7
	} else {
		disconnect(uiTlmb.tlmb_cb_plAddTo_artist, SIGNAL(currentIndexChanged(int)), this, SLOT(cbAddToPlArtistChgd(int)));
	}
}

void Tlmb::cbAddToPlArtistEnable(bool bD)			// enable (true) / disable (false) artist add-to playlist combobox
{
	if (bD == true) {
		uiTlmb.tlmb_cb_plAddTo_artist->setEnabled(true);
		cbAddToPlArtistCon(true);
	} else {
		cbAddToPlArtistCon(false);
		uiTlmb.tlmb_cb_plAddTo_artist->clear();
		uiTlmb.tlmb_cb_plAddTo_artist->setEnabled(false);
		if (iAddToPlCb == 7) {
			cbAddToPlSetCurrent();						// set current add-to playlist
		}
	}
}

void Tlmb::cbAddToPlArtistMenuCalled(const QPoint& pPosD)	// right click in artist add-to playlist combo box - call context menu
{
	pPos = pPosD;
	iPlCbCalled = 7;
	cbAddToPlMenuShow();
}

// **********
void Tlmb::cbAddToPlClearAll()						// clear all playlists add-to comboboxes, e. g. when new root dir
{
	cbAddToPlArtistCon(false);
	uiTlmb.tlmb_cb_plAddTo_artist->clear();

	cbAddToPlMusicTypeCon(false);
	uiTlmb.tlmb_cb_plAddTo_musicType->clear();

	cbAddToPlRootCon(false);
	uiTlmb.tlmb_cb_plAddTo_root->clear();
	sAddToPl = "";
	iAddToPlCb = 0;
}

void Tlmb::cbAddToPlMenuShow()						// show context menu for a add-to playlist combobox
{
//cout << "cbPlAddToMenuCalled - iCbPlActive = " << QString::number(iCbPlActive).toStdString() << endl;			// ***************  debug  ***************
	menuCbPlAddTo->exec(QCursor::pos());
}

// **********
void Tlmb::cbAddToPlMusicTypeChgd(int iD)			// current index of add-to music type combobox has changed
{
	iAddToPlCbCurIdx = iD;
	iAddToPlCb = 6;
	sAddToPl = uiTlmb.tlmb_cb_plAddTo_musicType->currentText();
	plAddToSetNew(sAddToPl);
}

void Tlmb::cbAddToPlMusicTypeCon(bool bD)			// connect add-to music type playlist combobox signals
{
	if (bD == true) {
		connect(uiTlmb.tlmb_cb_plAddTo_musicType, SIGNAL(currentIndexChanged(int)), this, SLOT(cbAddToPlMusicTypeChgd(int)));	// set add-to playlist to this selection and iAddToPlCb to 6
	} else {
		disconnect(uiTlmb.tlmb_cb_plAddTo_musicType, SIGNAL(currentIndexChanged(int)), this, SLOT(cbAddToPlMusicTypeChgd(int)));
	}
}

void Tlmb::cbAddToPlMusicTypeEnable(bool bD)		// enable (true) / disable (false) music type add-to playlist combobox
{
	if (bD == true) {
		uiTlmb.tlmb_cb_plAddTo_musicType->setEnabled(true);
		cbAddToPlMusicTypeCon(true);
	} else {
		cbAddToPlMusicTypeCon(false);
		uiTlmb.tlmb_cb_plAddTo_musicType->clear();
		uiTlmb.tlmb_cb_plAddTo_musicType->setEnabled(false);
	}
}

void Tlmb::cbAddToPlMusicTypeMenuCalled(const QPoint& pPosD)	// right click in music type add-to playlist combo box - call context menu
{
	pPos = pPosD;
	iPlCbCalled = 6;
	cbAddToPlMenuShow();
}

// **********
void Tlmb::cbAddToPlRootChgd(int iD)				// current index of add-to root combobox has changed
{
	iAddToPlCbCurIdx = iD;
	iAddToPlCb = 5;
	sAddToPl = uiTlmb.tlmb_cb_plAddTo_root->currentText();
	plAddToSetNew(sAddToPl);
}

void Tlmb::cbAddToPlRootCon(bool bD)				// connect add-to root playlist combobox signals
{
	if (bD == true) {
		connect(uiTlmb.tlmb_cb_plAddTo_root, SIGNAL(currentIndexChanged(int)), this, SLOT(cbAddToPlRootChgd(int)));	// set add-to playlist to this selection and iAddToPlCb to 5
	} else {
		disconnect(uiTlmb.tlmb_cb_plAddTo_root, SIGNAL(currentIndexChanged(int)), this, SLOT(cbAddToPlRootChgd(int)));
	}
}

void Tlmb::cbAddToPlRootEnable(bool bD)				// enable (true) / disable (false) root add-to playlist combobox
{
	if (bD == true) {
		uiTlmb.tlmb_cb_plAddTo_root->setEnabled(true);
		cbAddToPlRootCon(true);
	} else {
		cbAddToPlRootCon(false);
		uiTlmb.tlmb_cb_plAddTo_root->clear();
		uiTlmb.tlmb_cb_plAddTo_root->setEnabled(false);
	}
}

void Tlmb::cbAddToPlRootMenuCalled(const QPoint& pPosD)	// right click in root add-to playlist combo box - call context menu
{
	pPos = pPosD;
	iPlCbCalled = 5;
	cbAddToPlMenuShow();
}

// **********
bool Tlmb::cbAddToPlSetCurrent() 					// set current add-to playlist
{
	if (iAddToPlCb > 4 and iAddToPlCb < 8) {
		bD = false;
		switch (iAddToPlCb) {
			case 5:										// add-to playlist root
				cbAddToPlArtistCon(false);
				iD = uiTlmb.tlmb_cb_plAddTo_root->findText(sAddToPl);
				if (iD > -1) {
					uiTlmb.tlmb_cb_plAddTo_root->setCurrentIndex(iD);
					bD = true;
				} else {
					uiTlmb.tlmb_cb_plAddTo_root->setCurrentIndex(0);
				}
				cbAddToPlArtistCon(true);
				break;
			case 6:										// add-to playlist music type
				cbAddToPlMusicTypeCon(false);
				iD = uiTlmb.tlmb_cb_plAddTo_musicType->findText(sAddToPl);
				if (iD > -1) {
					uiTlmb.tlmb_cb_plAddTo_musicType->setCurrentIndex(iD);
					bD = true;
				} else {
					uiTlmb.tlmb_cb_plAddTo_musicType->setCurrentIndex(0);
				}
				cbAddToPlMusicTypeCon(true);
				break;
			case 7:										// add-to playlist artist
				cbAddToPlRootCon(false);
				iD = uiTlmb.tlmb_cb_plAddTo_artist->findText(sAddToPl);
				if (iD > -1) {
					uiTlmb.tlmb_cb_plAddTo_artist->setCurrentIndex(iD);
					bD = true;
				} else {
					uiTlmb.tlmb_cb_plAddTo_artist->setCurrentIndex(0);
				}
				cbAddToPlRootCon(true);
				break;
		}
	
		if (bD == false) {
			mbTxt_AddToPlNotFound();
			sAddToPl = "";
			iAddToPlCb = 0;
		}
	}													// we fail silently if this function was called out off range, e. g. 0 = none add-to playlist set
	return bD;
}

// **************************************
// **  disable combobox(es) functions  **
// **************************************
void Tlmb::cbDisableAll()							// disable and clear all comboboxes and set dir & name strings to ""
{
	cbPlayArtistEnable(false);
	cbPlayMusicTypeEnable(false);
	
	cbPlayPlArtistEnable(false);
	cbPlayPlMusicTypeEnable(false);
	cbPlayPlRootEnable(false);
	
	cbAddToPlArtistEnable(false);
	cbAddToPlMusicTypeEnable(false);
	cbAddToPlRootEnable(false);
	
	twSongsClear();

//	iPlayCb = 0;
	iAddToPlCb = 0;
	sPlayPl = "";
	sPlayPlFull = "";
	sPlaySong = "";
}

void Tlmb::cbDisableArtist()						// disable and clear all artist comboboxes and set dir & name string to ""
{
	if (iPlayCb == 1 or iPlayCb == 4) {					// playing from artist or artist playlist combobox, clear songs widget
		twSongsClear();
		sPlaySong = "";
	}
	cbPlayArtistEnable(false);
	cbPlayPlArtistEnable(false);
	cbAddToPlArtistEnable(false);
	if (iAddToPlCb == 7) {
		cbAddToPlSetCurrent();							// set current add-to playlist
	}

//	iPlayCb = 0;
//	iAddToPlCb = 0;
//	sPlayPl = "";
}

// ***************************************	
// **  indicator icon handling 	    	**
// ***************************************
void Tlmb::cbIndicatorCreate()						// load and scale indicator icons
{
	pmPlaying = QPixmap("/usr/share/tlmb/icons/scalable/music-note-blue.svg").scaled(iIconSize,iIconSize);
	pmPlAddTo = QPixmap("/usr/share/tlmb/icons/scalable/plus-green.svg").scaled(iIconSize,iIconSize);
}

void Tlmb::cbIndicatorPlayingClear()				// clear playing indicator for all 4 possible comboboxes
{
	uiTlmb.tlmb_l_playing_artist->clear();
	uiTlmb.tlmb_l_playing_artist->setToolTip("");
	uiTlmb.tlmb_l_playing_rootPl->clear();
	uiTlmb.tlmb_l_playing_rootPl->setToolTip("");
	uiTlmb.tlmb_l_playing_musicTypePl->clear();
	uiTlmb.tlmb_l_playing_musicTypePl->setToolTip("");
	uiTlmb.tlmb_l_playing_artistPl->clear();
	uiTlmb.tlmb_l_playing_artistPl->setToolTip("");
}

void Tlmb::cbIndicatorPlayingSet()					// set playing indicator for combobox playing songs from
{
	cbIndicatorPlayingClear();
	sTT = tr("Shown songs are from this combobox");
	switch (iPlayCb) {										// set combobox last played from
		case 1:														// artist/band
			if (uiTlmb.tlmb_cb_artist->isEnabled() == true) {
			    uiTlmb.tlmb_l_playing_artist->setPixmap(pmPlaying);
			    uiTlmb.tlmb_l_playing_artist->setToolTip(sTT);
			}
		    break;
		case 2:														// root playlists
			if (uiTlmb.tlmb_cb_root_playlists->isEnabled() == true) {
			    uiTlmb.tlmb_l_playing_rootPl->setPixmap(pmPlaying);
			    uiTlmb.tlmb_l_playing_rootPl->setToolTip(sTT);
			}
		    break;
		case 3:														// music type playlists
			if (uiTlmb.tlmb_cb_musicType_playlists->isEnabled() == true) {
			    uiTlmb.tlmb_l_playing_musicTypePl->setPixmap(pmPlaying);
			    uiTlmb.tlmb_l_playing_musicTypePl->setToolTip(sTT);
			}
		    break;
		case 4:														// artist/band playlists
			if (uiTlmb.tlmb_cb_artist_playlists->isEnabled() == true) {
			    uiTlmb.tlmb_l_playing_artistPl->setPixmap(pmPlaying);
			    uiTlmb.tlmb_l_playing_artistPl->setToolTip(sTT);
			}
		    break;
	}
}

void Tlmb::cbIndicatorPlAddToClear()				// clear current playlist indicator for all 3 possible playlist comboboxes
{
	uiTlmb.tlmb_l_plAddTo_root->clear();
	uiTlmb.tlmb_l_plAddTo_root->setToolTip("");
	uiTlmb.tlmb_l_plAddTo_musicType->clear();
	uiTlmb.tlmb_l_plAddTo_musicType->setToolTip("");
	uiTlmb.tlmb_l_plAddTo_artist->clear();
	uiTlmb.tlmb_l_plAddTo_artist->setToolTip("");
}

void Tlmb::cbIndicatorPlAddToSet()					// set indicator for playlist combobox songs will be added to
{
	cbIndicatorPlAddToClear();
	switch (iAddToPlCb) {										// set combobox last played from
		case 5:														// root playlists
			if (uiTlmb.tlmb_cb_plAddTo_root->isEnabled() == true) {
			    uiTlmb.tlmb_l_plAddTo_root->setPixmap(pmPlAddTo);
			}
		    break;
		case 6:														// music type playlists
			if (uiTlmb.tlmb_cb_plAddTo_musicType->isEnabled() == true) {
			    uiTlmb.tlmb_l_plAddTo_musicType->setPixmap(pmPlAddTo);
			}
		    break;
		case 7:														// artist/band playlists
			if (uiTlmb.tlmb_cb_plAddTo_artist->isEnabled() == true) {
			    uiTlmb.tlmb_l_plAddTo_artist->setPixmap(pmPlAddTo);
			}
		    break;
	} 
}

// **********************************************************
// **  common play and add-to playlist combobox functions  **
// **********************************************************
void Tlmb::cbPlPopArtist()							// populate artist playlists comboboxes - play and add-to
{
	slPlArtist = rwGetListPlaylists(pathGetAllowedPl(4));	// playlists in selected artist dir
	if (slPlArtist.count() > 0) {
		cbPlayPlArtistCon(false);
		uiTlmb.tlmb_cb_artist_playlists->setEnabled(true);
		uiTlmb.tlmb_cb_artist_playlists->clear();
		uiTlmb.tlmb_cb_artist_playlists->insertItems(0, slPlArtist);
		uiTlmb.tlmb_cb_artist_playlists->setToolTip(tr("Select existing playlist to play from for the selected artist/band"));
		cbPlayPlArtistCon(true);

		cbAddToPlArtistCon(false);
		uiTlmb.tlmb_cb_plAddTo_artist->setEnabled(true);
		uiTlmb.tlmb_cb_plAddTo_artist->clear();
		uiTlmb.tlmb_cb_plAddTo_artist->insertItems(0, slPlArtist);
		uiTlmb.tlmb_cb_plAddTo_artist->insertItem(0, "");
		uiTlmb.tlmb_cb_plAddTo_artist->setCurrentIndex(0);
		uiTlmb.tlmb_cb_plAddTo_artist->setToolTip(tr("Select existing playlist to add songs to for the selected artist/band"));
		cbAddToPlArtistCon(true);
	} else {
		cbPlayPlArtistCon(false);
		uiTlmb.tlmb_cb_artist_playlists->clear();
		if (iPlayCb == 4) {
			iPlayCb = 0;
		}
		
		cbAddToPlArtistCon(false);
		uiTlmb.tlmb_cb_plAddTo_artist->clear();
		if (iAddToPlCb == 7) {
			iAddToPlCb = 0;
		}
	}
}

void Tlmb::cbPlPopMusicType()						// populate music type playlists comboboxes - play and add-to
{
	slPlMusicType = rwGetListPlaylists(pathGetAllowedPl(3));	// playlists in selected music type dir
	if (slPlMusicType.count() > 0) {
		cbPlayPlMusicTypeCon(false);
		uiTlmb.tlmb_cb_musicType_playlists->clear();
		uiTlmb.tlmb_cb_musicType_playlists->insertItems(0, slPlMusicType);
		uiTlmb.tlmb_cb_musicType_playlists->setToolTip(tr("Select existing playlist to play from for the selected music type and across all artists and bands"));
		cbPlayPlMusicTypeCon(true);

		cbAddToPlMusicTypeCon(false);
		uiTlmb.tlmb_cb_plAddTo_musicType->clear();
		uiTlmb.tlmb_cb_plAddTo_musicType->insertItems(0, slPlMusicType);
		uiTlmb.tlmb_cb_plAddTo_musicType->insertItem(0, "");
		uiTlmb.tlmb_cb_plAddTo_musicType->setCurrentIndex(0);
		uiTlmb.tlmb_cb_plAddTo_musicType->setToolTip(tr("Select existing playlist to add songs to for the selected music type and across all artists and bands"));
		cbAddToPlMusicTypeCon(true);
	} else {
		cbPlayPlMusicTypeCon(false);
		uiTlmb.tlmb_cb_musicType_playlists->clear();
		if (iPlayCb == 3) {
			iPlayCb = 0;
		}

		cbAddToPlMusicTypeCon(false);
		uiTlmb.tlmb_cb_plAddTo_musicType->clear();
		if (iAddToPlCb == 6) {
			iAddToPlCb = 0;
		}
	}
}

void Tlmb::cbPlPopRoot()                     		// populate root path playlists comboboxes - play and add-to
{
	slPlRoot = rwGetListPlaylists(pathGetAllowedPl(2));	// playlists in root/playlists dir
	if (slPlRoot.count() > 0) {
		cbPlayPlRootCon(false);
		uiTlmb.tlmb_cb_root_playlists->clear();
		uiTlmb.tlmb_cb_root_playlists->insertItems(0, slPlRoot);
		uiTlmb.tlmb_cb_root_playlists->setToolTip(tr("Select existing playlist to play from for music files across all music types and artists/bands"));
		cbPlayPlRootCon(true);

		cbAddToPlRootCon(false);
		uiTlmb.tlmb_cb_plAddTo_root->clear();
		uiTlmb.tlmb_cb_plAddTo_root->insertItems(0, slPlRoot);
		uiTlmb.tlmb_cb_plAddTo_root->insertItem(0, "");
		uiTlmb.tlmb_cb_plAddTo_root->setCurrentIndex(0);
		uiTlmb.tlmb_cb_plAddTo_root->setToolTip(tr("Select existing playlist to add songs to for music files across all music types and artists/bands"));
		cbAddToPlRootCon(true);
	} else {
		cbPlayPlRootCon(false);
		uiTlmb.tlmb_cb_root_playlists->clear();
		if (iPlayCb == 2) {
			iPlayCb = 0;
		}

		cbAddToPlRootCon(false);
		uiTlmb.tlmb_cb_plAddTo_root->clear();
		if (iAddToPlCb == 5) {
			iAddToPlCb = 0;
		}
	}
}

void Tlmb::cbPlPopUpd(int iD)						// update playlist comboboxes for iAddToPlCB level - artists, music tyep, root
{
	switch (iD) {
		case 2:												// root level
			cbPlPopRoot();
			break;
		case 3:												// music type level
			cbPlPopMusicType();
			break;
		case 4:												// artist/band level
			cbPlPopArtist();
			break;
		case 5:												// root level
			cbPlPopRoot();
			break;
		case 6:												// music type level
			cbPlPopMusicType();
			break;
		case 7:												// artist/band level
			cbPlPopArtist();
			break;
	}
}

void Tlmb::cbPlayChgd()								// the combobox playing from has changed - needed?
{
	settingsWrite();
	twSongsChk();
}

// ********************************************************
// **  playing music type and artist combobox functions  **
// ********************************************************
void Tlmb::cbPlayArtistActivated(int iD)			// set combobox to play from to Artist/band, iPlayCb = 1
{
	cbPlayArtistChgd();
}

void Tlmb::cbPlayArtistChgd()						// artist combobox index changed
{
	iPlayCb = 1;
	sPlayArtist = uiTlmb.tlmb_cb_artist->currentText();
//	sPlayPlFull = pathGetAllowedPl(iPlayCb) + "/" + sPlayPl;
	if (iPlayCb == 4) {									// were playing from artist playlist - re-set to nil
		iPlayCb = 0;
		cbIndicatorPlayingClear();
	}
	
	if (iAddToPlCb == 7) {								// add-to playlist was artist playlist - re-set to nil
		iAddToPlCb = 0;
		cbIndicatorPlAddToClear();
	}
	cbPlPopArtist();									// populate artist playlists comboboxes - play and add-to
	
	if (updLevelArtist(true) == true) {
		iPlayCb = 1;
		updLevelSong(true);
	} else {
		updLevelSong(false);
	}
}

void Tlmb::cbPlayArtistClear()						// clear artist combobox
{
	cbPlayArtistCon(false);
	uiTlmb.tlmb_cb_artist->clear();
//    uiTlmb.tlmb_cb_artist->setEnabled(false);
    slPlayArtists.clear();
}

void Tlmb::cbPlayArtistCon(bool bD)					// connect (true) / disconnect (false) artist/band combobox signals
{
	if (bD == true) {
		connect(uiTlmb.tlmb_cb_artist, SIGNAL(activated(int)), this, SLOT(cbPlayArtistActivated(int)));		// set combobox to play from to 1
		connect(uiTlmb.tlmb_cb_artist, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayArtistChgd()));	// artist combobox index changed
	} else {
		disconnect(uiTlmb.tlmb_cb_artist, SIGNAL(activated(int)), this, SLOT(cbPlayArtistActivated(int)));		// set combobox to play from to 1
		disconnect(uiTlmb.tlmb_cb_artist, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayArtistChgd()));	// artist combobox index changed
	}
}

void Tlmb::cbPlayArtistEnable(bool bD)				// enable (true) / disable (false) artist play combobox
{
	if (bD == true) {
		cbPlayArtistCon(true);
	} else {
		cbPlayArtistCon(false);
		uiTlmb.tlmb_cb_artist->clear();
		uiTlmb.tlmb_cb_artist->setEnabled(false);
		if (iPlayCb == 1) {
			iPlayCb = 0;
		}
		sPlayArtist = "";
	}
}

void Tlmb::cbPlayArtistNone()						// no artists - disable all artists comboboxes and song widget
{
	mbTxt_ArtistNoneFound();
	uiTlmb.tlmb_cb_artist->setToolTip(tr("No artists found in the selected music type directory"));
	cbPlayArtistClear();
	cbPlayPlArtistClear();
	twSongsClear();
}

void Tlmb::cbPlayArtistPop()						// populate artist combobox
{
//	slPlayArtists = rwGetListDirs(sRootDir + "/" + uiTlmb.tlmb_cb_musicType->currentText() );    // artist dirs in selected music type dir
	if (slPlayArtists.count() > 0) {
		cbPlayArtistCon(false);
		uiTlmb.tlmb_cb_artist->setEnabled(true);
		uiTlmb.tlmb_cb_artist->clear();
		uiTlmb.tlmb_cb_artist->insertItems(0, slPlayArtists);
		uiTlmb.tlmb_cb_artist->setToolTip(tr("Select an available artist"));
		cbPlayArtistCon(true);
	} else {
		updLevelArtist(false);
	}
}

bool Tlmb::cbPlayArtistSetCurrent(const QString& sD)	// set current artist/band
{
	if (uiTlmb.tlmb_cb_artist->count() > 0) {			// just be save
		cbPlayArtistCon(false);
		iD = uiTlmb.tlmb_cb_artist->findText(sPlayArtist);
		if (iD > -1) {
			uiTlmb.tlmb_cb_artist->setCurrentIndex(iD);
		} else {
			uiTlmb.tlmb_cb_artist->setCurrentIndex(0);
		}
		sPlayArtist = uiTlmb.tlmb_cb_artist->currentText();
		cbPlayArtistCon(true);
		return true;
	} else {
		sPlayArtist = "";
		return false;
	}
}

// *****
void Tlmb::cbPlayMusicTypeChgd()					// music type combobox index changed
{
	sPlayMusicType = uiTlmb.tlmb_cb_musicType->currentText();
	if (iPlayCb > 2 and iPlayCb < 5) {					// were playing from playlist either music type or artist - re-set to nil
		iPlayCb = 0;
		cbIndicatorPlayingClear();
	}
	if (iAddToPlCb > 5 and iAddToPlCb < 8) {			// add-to playlist was either music type or artist - re-set to nil
		iAddToPlCb = 0;
		cbIndicatorPlAddToClear();
	}
	cbPlPopMusicType();									// populate music type playlists comboboxes - play and add-to

	if (updLevelArtist(true) == true) {
		iPlayCb = 1;									// start playing from artist/band dir
		if (updLevelSong(true) == true) {
		} else {
		}
	} else {
//cout << "cbPlayMusicTypeChgd - updLevelArtist(true) = false" << endl;                                               // *****  debug  *****
		cbPlayArtistNone();						// no artists, clear/ disable corresponding widgets
	}
}

bool Tlmb::cbPlayMusicTypeChk()						// check for music type dirs in root dir, once during start - disable all if none
{
	slPlayMusicTypes = rwGetListDirs(sRootDir);
	if (slPlayMusicTypes.count() > 0) {
		cbPlayMusicTypePop();
		return true;
    } else {
		cbPlayMusicTypeNone();
		return false;
	}
}

void Tlmb::cbPlayMusicTypeClear()					// clear music type combobox
{
	cbPlayMusicTypeCon(false);
	uiTlmb.tlmb_cb_musicType->clear();
//    uiTlmb.tlmb_cb_musicType->setEnabled(false);
    uiTlmb.tlmb_cb_musicType->setToolTip(tr("No music type directory found - see the online help, Ctrl + h, for the required music file structure"));
    slPlayMusicTypes.clear();
    sPlayMusicType = "";
    bMusicType = false;										// needed?
}

void Tlmb::cbPlayMusicTypeCon(bool bD)				// connect music type combobox signals
{
	if (bD == true) {
		connect(uiTlmb.tlmb_cb_musicType, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayMusicTypeChgd()));	// music type combobox index changed
	} else {
		disconnect(uiTlmb.tlmb_cb_musicType, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayMusicTypeChgd()));	// music type combobox index changed
	}
}

void Tlmb::cbPlayMusicTypeEnable(bool bD)			// enable (true) / disable (false) music type play combobox
{
	if (bD == true) {
		cbPlayMusicTypeCon(true);
	} else {
		cbPlayMusicTypeCon(false);
		uiTlmb.tlmb_cb_musicType->clear();
		uiTlmb.tlmb_cb_musicType->setEnabled(false);
		sPlayMusicType = "";
	}
}

void Tlmb::cbPlayMusicTypeNone()					// no music types - disable all relevant combo boxes
{
	mbTxt_MusicTypeNoneFound();
	cbPlayMusicTypeClear();
	cbPlayPlMusicTypeClear();
	cbPlayArtistClear();
	cbPlayPlArtistClear();
	twSongsClear();
	uiTlmb.tlmb_cb_plAddTo_musicType->clear();
	uiTlmb.tlmb_cb_plAddTo_artist->clear();
	
}

void Tlmb::cbPlayMusicTypePop()						// populate music type comboboxes,  play and add-to
{
	if (slPlayMusicTypes.count() > 0) {
		cbPlayMusicTypeCon(false);
		uiTlmb.tlmb_cb_musicType->setEnabled(true);
		uiTlmb.tlmb_cb_musicType->clear();
		uiTlmb.tlmb_cb_musicType->insertItems(0, slPlayMusicTypes);
		uiTlmb.tlmb_cb_musicType->setToolTip(tr("Select an available music type / genre"));
		bMusicType = true;										// needed?
		cbPlayMusicTypeCon(true);
    } else {
		cbPlayMusicTypeNone();
	}
}

bool Tlmb::cbPlayMusicTypeSetCurrent(const QString& sD)	// set current music type
{
	if (uiTlmb.tlmb_cb_musicType->count() > 0) {			// just be save
		cbPlayMusicTypeCon(false);
		iD = uiTlmb.tlmb_cb_musicType->findText(sD);
		if (iD > -1) {
			uiTlmb.tlmb_cb_musicType->setCurrentIndex(iD);
		} else {
			uiTlmb.tlmb_cb_musicType->setCurrentIndex(0);
		}
		sPlayMusicType = uiTlmb.tlmb_cb_musicType->currentText();
		cbPlayMusicTypeCon(true);
		return true;
	} else {
		sPlayMusicType = "";
		return false;
	}
}

// **********
void Tlmb::cbPlayPlArtistActivated(int iD)			// set combobox to play from to artist/band playlists, iPlayCb = 4
{
	cbPlayPlArtistChgd();
}

void Tlmb::cbPlayPlArtistChgd()						// play artist combobox index changed
{
	iPlayCb = 4;
	sPlayPlArtist = uiTlmb.tlmb_cb_artist_playlists->currentText();
	sPlayPl = sPlayPlArtist;
	sPlayPlFull = pathGetAllowedPl(iPlayCb) + "/" + sPlayPl;
	updLevelSong(true);
}

void Tlmb::cbPlayPlArtistClear()					// disable artist playlists combobox
{
	cbPlayPlArtistCon(false);
	uiTlmb.tlmb_cb_artist_playlists->clear();
    uiTlmb.tlmb_cb_artist_playlists->setEnabled(false);
    uiTlmb.tlmb_cb_artist_playlists->setToolTip(tr("No playlists found for the selected artist"));
//    slPlayPlArtist.clear();
}

void Tlmb::cbPlayPlArtistCon(bool bD)				// connect artist/band playlist combobox signals
{
	if (bD == true) {
		connect(uiTlmb.tlmb_cb_artist_playlists, SIGNAL(activated(int)), this, SLOT(cbPlayPlArtistActivated(int)));	// set combobox to play from to 4
		connect(uiTlmb.tlmb_cb_artist_playlists, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayPlArtistChgd()));	// artist playlists combobox index changed
	} else {
		disconnect(uiTlmb.tlmb_cb_artist_playlists, SIGNAL(activated(int)), this, SLOT(cbPlayPlArtistActivated(int)));
		disconnect(uiTlmb.tlmb_cb_artist_playlists, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayPlArtistChgd()));
	}
}

void Tlmb::cbPlayPlArtistEnable(bool bD)			// enable (true) / disable (false) artist play playlists combobox
{
	if (bD == true) {
		cbPlayPlArtistCon(true);
	} else {
		cbPlayPlArtistCon(false);
		uiTlmb.tlmb_cb_artist_playlists->clear();
		uiTlmb.tlmb_cb_artist_playlists->setEnabled(false);
		if (iPlayCb == 4) {
			iPlayCb = 0;
		}
	}
}

void Tlmb::cbPlayPlArtistMenuCalled(const QPoint& pPosD)	// right click in artist playing playlist combo box - call context menu
{
	pPos = pPosD;
	iPlCbCalled = 4;
	cbPlayPlMenuShow();
}

bool Tlmb::cbPlayPlArtistSetCurrent(const QString& sD)	// set current artist/band playlist
{
	if (uiTlmb.tlmb_cb_artist_playlists->count() > 0) {			// just be save
		cbPlayPlArtistCon(false);
		iD = uiTlmb.tlmb_cb_artist_playlists->findText(sD);
		if (iD > -1) {
			uiTlmb.tlmb_cb_artist_playlists->setCurrentIndex(iD);
		} else {
			uiTlmb.tlmb_cb_artist_playlists->setCurrentIndex(0);
		}
		sPlayPlArtist = uiTlmb.tlmb_cb_artist_playlists->currentText();
		cbPlayPlArtistCon(true);
		return true;
	} else {
		sPlayPlArtist = "";
		return false;
	}
}

// *****
void Tlmb::cbPlayPlMenuShow()						// show context menu for a playing playlist combobox
{
//cout << "cbPlPlayMenuShow - iCbPlActive = " << QString::number(iCbPlActive).toStdString() << endl;			// ***************  debug  ***************
	menuCbPlPlaying->exec(QCursor::pos());
}

// *****
void Tlmb::cbPlayPlMusicTypeActivated(int iD)		// set combobox to play from to music type playlists, iPlayCb = 3
{
	cbPlayPlMusicTypeChgd();
}

void Tlmb::cbPlayPlMusicTypeChgd()					// play music type combobox index changed
{
	iPlayCb = 3;
	sPlayPlMusicType = uiTlmb.tlmb_cb_musicType_playlists->currentText();
	sPlayPl = sPlayPlMusicType;
	sPlayPlFull = pathGetAllowedPl(iPlayCb) + "/" + sPlayPl;
//cout << "cbPlayPlmusicTypeChgd: reached" << endl;                                               // *****  debug  *****
	updLevelSong(true);
}

void Tlmb::cbPlayPlMusicTypeClear()					// clear and disable music type playlists combobox
{
	cbPlayPlMusicTypeCon(false);
	uiTlmb.tlmb_cb_musicType_playlists->clear();
    uiTlmb.tlmb_cb_musicType_playlists->setEnabled(false);
    uiTlmb.tlmb_cb_musicType_playlists->setToolTip(tr("No playlists found for the selected music type"));
//    slPlayPlMusicType.clear();
}

void Tlmb::cbPlayPlMusicTypeCon(bool bD)			// connect music type playlist combobox signals
{
	if (bD == true) {
		connect(uiTlmb.tlmb_cb_musicType_playlists, SIGNAL(activated(int)), this, SLOT(cbPlayPlMusicTypeActivated(int)));	// set combobox to play from to 3
		connect(uiTlmb.tlmb_cb_musicType_playlists, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayPlMusicTypeChgd()));	// music type playlists combobox index changed
	} else {
		disconnect(uiTlmb.tlmb_cb_musicType_playlists, SIGNAL(activated(int)), this, SLOT(cbPlayPlMusicTypeActivated(int)));
		disconnect(uiTlmb.tlmb_cb_musicType_playlists, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayPlMusicTypeChgd()));
	}
}

void Tlmb::cbPlayPlMusicTypeEnable(bool bD)			// enable (true) / disable (false) music type play playlists combobox
{
	if (bD == true) {
		cbPlayPlMusicTypeCon(true);
	} else {
		cbPlayPlMusicTypeCon(false);
		uiTlmb.tlmb_cb_musicType_playlists->clear();
		uiTlmb.tlmb_cb_musicType_playlists->setEnabled(false);
		if (iPlayCb == 3) {
			iPlayCb = 0;
		}
	}
}

void Tlmb::cbPlayPlMusicTypeMenuCalled(const QPoint& pPosD)	// right click in music type playing playlist combo box - call context menu
{
	pPos = pPosD;
	iPlCbCalled = 3;
	cbPlayPlMenuShow();
}

bool Tlmb::cbPlayPlMusicTypeSetCurrent(const QString& sD)	// set current music type playlist
{
	if (uiTlmb.tlmb_cb_musicType_playlists->count() > 0) {			// just be save
		cbPlayPlMusicTypeCon(false);
		iD = uiTlmb.tlmb_cb_musicType_playlists->findText(sD);
		if (iD > -1) {
			uiTlmb.tlmb_cb_musicType_playlists->setCurrentIndex(iD);
		} else {
			uiTlmb.tlmb_cb_musicType_playlists->setCurrentIndex(0);
		}
		sPlayPlMusicType = uiTlmb.tlmb_cb_musicType_playlists->currentText();
		cbPlayPlMusicTypeCon(true);
		return true;
	} else {
		sPlayPlMusicType = "";
		return false;
	}
}

// *****
void Tlmb::cbPlayPlRootActivated(int iD)			// set combobox to play from to root playlists, iPlayCb = 2
{
	cbPlayPlRootChgd();
}

void Tlmb::cbPlayPlRootChgd()						// play root combobox index changed
{
	iPlayCb = 2;
	sPlayPlRoot = uiTlmb.tlmb_cb_root_playlists->currentText();
	sPlayPl = sPlayPlRoot;
	sPlayPlFull = pathGetAllowedPl(iPlayCb) + "/" + sPlayPl;
//cout << "cbPlayPlRootChgd: reached" << endl;                                               // *****  debug  *****
	updLevelSong(true);
}

void Tlmb::cbPlayPlRootCon(bool bD)					// connect root playlist combobox signals
{
	if (bD == true) {
		connect(uiTlmb.tlmb_cb_root_playlists, SIGNAL(activated(int)), this, SLOT(cbPlayPlRootActivated(int)));	// set combobox to play from to 2
		connect(uiTlmb.tlmb_cb_root_playlists, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayPlRootChgd()));	// root playlists combobox index changed
	} else {
		disconnect(uiTlmb.tlmb_cb_root_playlists, SIGNAL(activated(int)), this, SLOT(cbPlayPlRootActivated(int)));
		disconnect(uiTlmb.tlmb_cb_root_playlists, SIGNAL(currentIndexChanged(int)), this, SLOT(cbPlayPlRootChgd()));
	}
}

void Tlmb::cbPlayPlRootEnable(bool bD)				// enable (true) / disable (false) root play playlists combobox
{
	if (bD == true) {
		cbPlayPlRootCon(true);
	} else {
		cbPlayPlRootCon(false);
		uiTlmb.tlmb_cb_root_playlists->clear();
		uiTlmb.tlmb_cb_root_playlists->setEnabled(false);
		if (iPlayCb == 2) {
			iPlayCb = 0;
		}
	}
}

void Tlmb::cbPlayPlRootMenuCalled(const QPoint& pPosD)	// right click in root playing playlist combo box - call context menu
{
	pPos = pPosD;
	iPlCbCalled = 2;
	cbPlayPlMenuShow();
}

bool Tlmb::cbPlayPlRootSetCurrent(const QString& sD) 	// set current root playlist
{
	if (uiTlmb.tlmb_cb_root_playlists->count() > 0) {	// just be save
		cbPlayPlRootCon(false);
		iD = uiTlmb.tlmb_cb_root_playlists->findText(sD);
		if (iD > -1) {
			uiTlmb.tlmb_cb_root_playlists->setCurrentIndex(iD);
		} else {
			uiTlmb.tlmb_cb_root_playlists->setCurrentIndex(0);
		}
		sPlayPlRoot = uiTlmb.tlmb_cb_root_playlists->currentText();
		cbPlayPlRootCon(true);
		return true;
	} else {
		sPlayPlRoot = "";
		return false;
	}
}

// *****
void Tlmb::cbPlaySet()								// set current playing from combobox
{
	if (iPlayCb == 0) {
		iPlayCb = 1;									// set artist combobox to play from if we do not having anything else
	}
	switch (iPlayCb) {
			case 1:											// artist dir
				uiTlmb.tlmb_cb_artist->setFocus();
				break;
			case 2:											// root playlists
				uiTlmb.tlmb_cb_root_playlists->setFocus();
				break;
			case 3:											// music type playlists
				uiTlmb.tlmb_cb_musicType_playlists->setFocus();
				break;
			case 4:											// artist playlists
				uiTlmb.tlmb_cb_root_playlists->setFocus();
				break;
	}
	cbIndicatorPlayingSet();
}

int Tlmb::cbPop()									// populate all comboboxes - 0=success, 1=failure
{
	if (updLevelRoot() == true) {						// root dir is valid (exists and readable) and root widgets are successfully updated
		if (updLevelMusicType() == true) {					// we have music type dirs and the music type widgets are successfully updated
			if (updLevelArtist(true) == true) {					// we have artist dirs and the artist widgets are successfully updated
				return 0;
			}
		}
	}

//cout << "cbPop: " << endl;																									// *****  debug  *****

			return 1;
}

// **********************************************
// **  general custom menu functions           **
// **********************************************
void Tlmb::customMenusSetPolicy()					// set context menu policy for widgets with custom context menu
{
	uiTlmb.tlmb_cb_root_playlists->setContextMenuPolicy(Qt::CustomContextMenu);
	uiTlmb.tlmb_cb_musicType_playlists->setContextMenuPolicy(Qt::CustomContextMenu);
	uiTlmb.tlmb_cb_artist_playlists->setContextMenuPolicy(Qt::CustomContextMenu);
	uiTlmb.tlmb_tw_Songs->setContextMenuPolicy(Qt::CustomContextMenu);
	uiTlmb.tlmb_cb_plAddTo_root->setContextMenuPolicy(Qt::CustomContextMenu);
	uiTlmb.tlmb_cb_plAddTo_musicType->setContextMenuPolicy(Qt::CustomContextMenu);
	uiTlmb.tlmb_cb_plAddTo_artist->setContextMenuPolicy(Qt::CustomContextMenu);
}

void Tlmb::customMenusCon(bool bD)					// connect 1 / disconnect 0 custom menus
{
	if (bD == true) {
		connect(uiTlmb.tlmb_tw_Songs, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(twSongsMenuCalled(const QPoint&)));
		connect(uiTlmb.tlmb_cb_root_playlists, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(cbPlayPlRootMenuCalled(const QPoint&)));
		connect(uiTlmb.tlmb_cb_musicType_playlists, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(cbPlayPlMusicTypeMenuCalled(const QPoint&)));
		connect(uiTlmb.tlmb_cb_artist_playlists, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(cbPlayPlArtistMenuCalled(const QPoint&)));
		connect(uiTlmb.tlmb_cb_plAddTo_root, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(cbAddToPlRootMenuCalled(const QPoint&)));
		connect(uiTlmb.tlmb_cb_plAddTo_musicType, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(cbAddToPlMusicTypeMenuCalled(const QPoint&)));
		connect(uiTlmb.tlmb_cb_plAddTo_artist, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(cbAddToPlArtistMenuCalled(const QPoint&)));
	} else {
		
	}
}

// ****************************************
// **  play directory functions          **
// ****************************************
bool Tlmb::dirArtistChgd()							// artist dir has changed - try re-loading artist playlists and songs in dir
{
	return false;
}

bool Tlmb::dirArtistChk()							// check if artist dir is valid: exists, readable and has artist dirs
{
	slPlayArtists = rwGetListDirs(pathGetAllowedDir(3));    // artist dirs in selected music type dir
	if (slPlayArtists.count() > 0) {
//cout << "dirArtistChk: return true " << endl;                                               // *****  debug  *****
		return true;
	} else {
//cout << "dirArtistChk: return false " << endl;                                               // *****  debug  *****
		return false;
	}
}

bool Tlmb::dirMusicTypeChgd()						// music type dir has changed - try re-loading music playlists, artist dir and it's playlists and songs in artist dir
{
	return false;
}

bool Tlmb::dirMusicTypeChk()						// check if music type dir is valid: exists, readable and has music types dirs
{
	slPlayMusicTypes = rwGetListDirs(sRootDir);
	if (slPlayMusicTypes.count() > 0) {
		return true;
	} else {
		return false;
	}
}

bool Tlmb::dirRootChgd()							// root dir has changed - populate comboboxes and try re-loading everything
{
	if (cbPop() == 0) {
		cbPlaySet();
		if (twSongsChk() == true) {
			twSongsPop();
		}
//cout << "dirRootChgd: before cbAddToPlSetCurrent and cbIndictorPlAddToSet " << endl;                                               // *****  debug  *****
		cbAddToPlSetCurrent();
		cbIndicatorPlAddToSet();
		return true;
	} else {
		return false;									// levels failing have been disabled in cbPop() on error already
	}
}

bool Tlmb::dirRootChk()								// check if root dir is valid: exists and is readable
{
	if (sRootDir == "" or rwStatus(sRootDir) < 110) {
		return false;
	} else {
		return true;
	}
}

// ***********************************************************
// **  key press event handling                             **
// ***********************************************************
void Tlmb::keyPressEvent(QKeyEvent *ke)				// key press event
{
//cout << "Tlmb::keyPressEvent(): reached !" << endl;                                               // *****  debug  *****
	switch (ke->keyCombination()) {
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F1):	// open music type combobox
//			uiTlmb.tlmb_cb_musicType->setFocus();
			uiTlmb.tlmb_cb_musicType->showPopup();
			break;
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F2):	// open artist type combobox
			uiTlmb.tlmb_cb_artist->showPopup();
			break;
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F3):	// open root playlists combobox
			uiTlmb.tlmb_cb_root_playlists->showPopup();
			break;
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F4):	// open music type playlists combobox
			uiTlmb.tlmb_cb_musicType_playlists->showPopup();
			break;
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F5):	// open artist playlists combobox
			uiTlmb.tlmb_cb_artist_playlists->showPopup();
			break;
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F6):	// open root add to playlists combobox
			uiTlmb.tlmb_cb_plAddTo_root->showPopup();
			break;
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F7):	// open music type add to playlists combobox
			uiTlmb.tlmb_cb_plAddTo_musicType->showPopup();
			break;
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F8):	// open artist add to playlists combobox
			uiTlmb.tlmb_cb_plAddTo_artist->showPopup();
			break;
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F9):	// set focus to songs table widget
			uiTlmb.tlmb_tw_Songs->setFocus();
			break;
		case Qt::Key_MediaPlay:									// play selected song
			playerStart();
			break;
//		case Qt::Key_MediaTogglePlayPause:						// play selected song
//			playerStart();
//			break;
		case QKeyCombination(Qt::ShiftModifier, Qt::Key_F12):	// add song to add-to playlist, table widget must have focus
//cout << "Tlmb::keyPressEvent(): Key_plus reached !" << endl;                                               // *****  debug  *****
			if (uiTlmb.tlmb_tw_Songs->hasFocus() == true) {
				plAddToAddSong();
			}
			break;
	}
}

// ************************************************************
// **  meta data handling - need a qt media player for this  **
// ************************************************************
void Tlmb::metadataGet()							// get meta data from song
{
	if (bMdPlayer == false) {
		metadataPlayerCreate();
	}

	mdPlayer->setSource(QUrl::fromLocalFile(strSongD.sPathName));	// strSongD was already set in calling plAddToAddSong
}

void Tlmb::metadataPlayerCreate()					// create qt media player only to retrieve songs meta data from it
{
//cout << "mediaPlayerCreate: reached " << endl;                                               // *****  debug  *****
	mdPlayer = new QMediaPlayer;
	connect(mdPlayer, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)), this, SLOT(metadataPlayerStatusChgd(QMediaPlayer::MediaStatus)));
	bMdPlayer = true;
}

void Tlmb::metadataPlayerStatusChgd(QMediaPlayer::MediaStatus state)	// mdPlayer: media status changed - needed just to retrieve meta data?
{
	switch (state) {
		case QMediaPlayer::LoadedMedia:
			mdData = mdPlayer->metaData();
			bMdHave = true;
			plAddToAddSongFinish();
			break;
		case QMediaPlayer::InvalidMedia:
			bMdHave = false;
			plAddToAddSongFinish();										// only continue on these 2 states! (loadedMedia or InvalidMedia) Else we might add the song several times! e. g. loading... state
			break;
	}
}

void Tlmb::metadataPlayerDelete()					// delete qt media player (only used to retrieve meta data from song)
{
	if (bMdPlayer == true) { 
		mdPlayer->stop();
		disconnect(mdPlayer, SIGNAL(mediaStatusChanged(QMediaPlayer::MediaStatus)), this, SLOT(mdPlayerStatusChanged(QMediaPlayer::MediaStatus)));
		mdPlayer->deleteLater();
		bMdPlayer = false;
	}
}

// ****************************************
// **  path helper functions             **
// ****************************************
QString Tlmb::pathGetAllowedDir(int iD)				// get allowed directory path depending which level we are in: music type=2, artist=3, song=4
{
	switch (iD) {
		case 2:											// music type dir
			sPathAllowedDir = sRootDir;
		    break;
		case 3:											// artist dir
			sPathAllowedDir = sRootDir + "/" + sPlayMusicType;
		    break;
		case 4:											// song dir
			sPathAllowedDir = sRootDir + "/" + sPlayMusicType + "/" + sPlayArtist;
		    break;
		default:
			sPathAllowedDir = "";
		    break;
	}
	return sPathAllowedDir;
}

QString Tlmb::pathGetAllowedPl(int iD)				// get allowed playlist file depending which level the playlist is in: play: root=2, music type=3, artist=4 ; add-to: root=5, music type=6, artist=7
{
	switch (iD) {									// set allowed path for playlists - /playlists directory under root dir, each music type dir and each artist/band dir
		case 2:											// playlist in root dir
			sPathAllowedPl = sRootDir + "/playlists";
		    break;
		case 3:											// playlist in music typ
			sPathAllowedPl = sRootDir + "/" + sPlayMusicType + "/playlists";
		    break;
		case 4:											// playlist in artist/band
			sPathAllowedPl = sRootDir + "/" + sPlayMusicType + "/" + sPlayArtist + "/playlists";
		    break;
		case 5:											// playlist in root dir
			sPathAllowedPl = sRootDir + "/playlists";
		    break;
		case 6:											// playlist in music typ
			sPathAllowedPl = sRootDir + "/" + sPlayMusicType + "/playlists";
		    break;
		case 7:											// playlist in artist/band
			sPathAllowedPl = sRootDir + "/" + sPlayMusicType + "/" + sPlayArtist + "/playlists";
		    break;
	}
	return sPathAllowedPl;
}

// ******************************************
// **  player functions - external player  **
// ******************************************
void Tlmb::playerCreate()           				// create player qprocess, once at program start
{
	playerProc = new PlayerProc();
	connect(playerProc, SIGNAL(sigPpChkEnd(bool)), this, SLOT(playerAvail(bool)));								// return from checking if defined player is available
	connect(playerProc, SIGNAL(sigPpExited(int)), this, SLOT(playerEnded(int)));								// qprocess playing a song ended: 0=normally, 1=crash
	connect(this, SIGNAL(sigTlmbPlayerChkAvail(const QString&, const QStringList&)), playerProc, SLOT(ppChkAvail(const QString&, const QStringList&)));	// check if player sD can be found - slD=player options
	connect(this, SIGNAL(sigTlmbPlayerStart(const QString&)), playerProc, SLOT(ppProcPrep(const QString&)));	// start playing song with external player
}

void Tlmb::playerDisableStop()           			// disable to stop automatically playing next song when playing to end of list or looping through it
{
	bStopPlaying = false;								// false = play next song, true = do not play next song
}

void Tlmb::playerStart()           					// start player qprocess
{
	strSongD = liSongStr.at(iSongCurIdx);
//cout << "tlmb: playerStart - sPathName = " << strSongD.sPathName.toStdString() << endl;                               // *****  debug  *****
	emit sigTlmbPlayerStart(strSongD.sPathName);
}

void Tlmb::playerEnded(int iD) 						// player qprocess ended - 0=normal, 1=crash
{
	if (iD == 0) {									// normal exit
		switch (iPlayMode) {
		    case 1:										// only single song - do nothing ?
			    
			    break;
		    case 2:										// to end of song list - play next if any left
				if (bStopPlaying == false) {
					if (iSongCurIdx < iSongCnt - 1) {
						bStopPlaying = true;
						timerStopPlayingDisableStart();
						twSongsCellDoubleClicked(iSongCurIdx + 1, 0);	// play next song: (row, col)
					}
					break;
				}
			case 3:										// loop through song list - play next or first if at end of list
				if (bStopPlaying == false) {
					bStopPlaying = true;
					timerStopPlayingDisableStart();
					if (iSongCurIdx < iSongCnt - 1) {
						twSongsCellDoubleClicked(iSongCurIdx + 1, 0);	// play next song: (row, col)
					} else {
						twSongsCellDoubleClicked(0, 0);					// re-start playing first song: (row, col)
					}
					break;
				}
		}
	} else {										// crash
		
	}
}

void Tlmb::playerAvail(bool bD)						// receive result of checking if player is available
{
//cout << "playerAvail reached " << QString::number(iPlayers).toStdString() << endl;                                               // *****  debug
	if (bD == true) {									// player found 
		init();
	} else {
		mbTxt_PlayerNoneFound();
//		quitPrep();
	}
}

// ****************************************
// **  playlist functions                **
// ****************************************
void Tlmb::plAddToAddSong()							// add current song to current add-to playlist
{
	if (iAddToPlCb > 4 and iAddToPlCb < 8) {			// add-to playlists 5=root, 6=musictype, 7=artist/band
		strSongD = liSongStr.at(iSongCurIdx);
		if (m3uItemExists(strSongD.sPathName, liM3uStrAddTo) == false) {	// song (full path/name) not already in list - don't add it twice
			metadataGet();								// we may have to wait on loading the media first, before we can retrieve any meta data from it - will continue on return in plAddToAddSongFinish()
		} else {
			mbTxt_AddToPlAddSongExists();
		}
	}
}

void Tlmb::plAddToAddSongFinish()					// finish add current song to current add-to playlist - media has been loaded, chance to extract meta data
{
	if (bMdHave == true) {
		strM3uD.sSec = QString::number(mdData.value(QMediaMetaData::Duration).toInt() / 1000);
	} else {
		strM3uD.sSec = "";
	}
	strM3uD.sSongName = strSongD.sName;
	strM3uD.sSongPathName = strSongD.sPathName;
	liM3uStrAddTo.append(strM3uD);
//cout << "plAddToAddSong - before plAddToWrite " << QString::number(liM3uStrAddTo.count()).toStdString() << endl;                                               // *****  debug
	if (plWrite(sAddToPlFull, liM3uStrAddTo) == true) {
		mbTxt_AddToPlAddSongSuccess();			// yeah, we did it!
 	} else {									// error while writing m3u immediately - internal playlist will be set to nil ???
		mbTxt_AddToPlAddSongFailed();			// could not write add-to playlist ?
	}
}

bool Tlmb::plAddToLoad()							// load add-to playlist as StrM3u list
{
	if (plLoad(true, sAddToPlFull) == true) {
		liM3uStrAddTo = liM3uStr;
		return true;
	} else {
		liM3uStrAddTo.clear();
		return false;
	}
}

void Tlmb::plAddToSetNew(const QString& sD)			// add-to playlist has changed - set new playlist to add songs to - sD = name only, also from playlistset widget
{
	sAddToPl = sD;

	cbAddToPlRootCon(false);
	uiTlmb.tlmb_cb_plAddTo_root->setCurrentIndex(0);

	cbAddToPlMusicTypeCon(false);
	uiTlmb.tlmb_cb_plAddTo_musicType->setCurrentIndex(0);

	cbAddToPlArtistCon(false);
	uiTlmb.tlmb_cb_plAddTo_artist->setCurrentIndex(0);
	
// cout << "plAddToSetNew reached " << QString::number(iAddToPlCbCurIdx).toStdString() << endl;                                               // *****  debug

//	sAddToPlFull = pathGetAllowedPl(iAddToPlCb) + "/" + sAddToPl;
	if (sAddToPl > "") {
		plAddToSetPathName();

		if (plAddToLoad() == true) {
//		cout << "plAddToSetNew - iAddToPlCurIdx = " << QString::number(iAddToPlCbCurIdx).toStdString() << endl;                                               // *****  debug
			cbAddToPlSetCurrent();
		}
	} else {
		iAddToPlCb = 0;
	}

	cbAddToPlRootCon(true);
	cbAddToPlMusicTypeCon(true);
	cbAddToPlArtistCon(true);

	cbIndicatorPlAddToSet();							// includes clearing old indicator
}

void Tlmb::plAddToSetPathName()						// set sAddToPlFull - full path and name of current add-to playlist
{
	sAddToPlFull = pathGetAllowedPl(iAddToPlCb) + "/" + sAddToPl;
}

bool Tlmb::plLoad(bool bD, const QString& sD)		// load a m3u playlist - return true success, false error on open , call false=play true=add-to
{
// cout << "plLoad: reached " << endl;                                               // *****  debug  *****
	if (bD == false) {
		sD3 = tr("play");
	} else {
		sD3 = tr("add-to");
	}
	liM3uStr = m3uItemGetList(sD);
	if (liM3uStr.count() == 1) {					// check if error or only empty
		strM3uD = liM3uStr.at(0);
		if (strM3uD.sSongName.left(6) == "Error!") {
			mbTxt_PlLoadError(sD3, sD);
			liM3uStr.clear();
			return false;
		}
	}
	return true;
}

bool Tlmb::plPlayLoad()								// load play playlist as StrM3u list
{
	if (plLoad(false, pathGetAllowedPl(iPlayCb) + "/" + sPlayPl) == true) {
		liM3uStrPlay = liM3uStr;
		twSongsList2StrList();
		return true;
	} else {
		liM3uStrPlay.clear();
		return false;
	}
}

void Tlmb::plPlayUpd(const QList<StrM3u>& liM3uStr)	// current playing playlist has been updated - re-load song list
{
	
}

bool Tlmb::plWrite(const QString& sD, const QList<StrM3u>& liM3uStr)	// save playlist - full path & name, m3u struct list
{
//cout << "plWrite - reached - count liM3uStr = " << QString::number(liM3uStr.count()).toStdString() << endl;                                               // *****  debug
	if (m3uFileWrite(sD, liM3uStr) == true) {			// successfully saved the playlist - full path & name, m3u struct list
		return true;
	} else {
//		sM3u = "";										// ******************* need to do better error handling here - but should never happen?
		return false;
	}
}

//  **********
bool Tlmb::sessionLastReCreate()					// try re-creating last session settings - true/1 = success
{
	iErrLastSes = 0;															// no error

  // check play comboboxes
	if (uiTlmb.tlmb_cb_musicType->currentText() != sLastMusicType) {			// music type
		iErrLastSes = 1;															// fatal, cannot re-create anything
		return false;
	} else {
		if (uiTlmb.tlmb_cb_artist->currentText() != sLastArtist) {				// artist
			iErrLastSes += 10;														// bad, but we might have more music types (***** check if there are other music type entries? *****)
		}
		if (uiTlmb.tlmb_cb_root_playlists->currentText() != sLastPlRoot) {		// root playlist
			iErrLastSes += 100;
		}
		if (uiTlmb.tlmb_cb_musicType_playlists->currentText() != sLastPlMusicType) {	// music type playlist
			iErrLastSes += 1000;
		}
		if (uiTlmb.tlmb_cb_artist_playlists->currentText() != sLastPlArtist) {	// artist playlist
			iErrLastSes += 10000;
		}
	}

  // populate song widget, this also sets playing indicator
  	iPlayCb = iLastCbPlay;
  	if (iPlayCb > 1 and iPlayCb < 5) {					// playing a playlist
		sPlayPl = "";
		switch (iPlayCb) {
			case 2:											// root dir playlist
				if (iErrLastSes < 100) {
					sPlayPl = sLastPlRoot;
				}
				break;
			case 3:											// music type playlist
				if (iErrLastSes < 1000) {
					sPlayPl = sLastPlMusicType;
				}
				break;
			case 4:											// artist playlist
				if (iErrLastSes < 10000) {
					sPlayPl = sLastPlArtist;
				}
				break;
		}
		if (sPlayPl == "") {							// cannot load playlist to play
			return false;
		}
	}

  // set add-to combobox
	iAddToPlCb = iLastCbPlAddTo;
	cbAddToPlSetCurrent();
	cbIndicatorPlAddToSet();

	if (twSongsChk() == true) {							// populate song table view
		twSongsPop();
	}
	return true;
}

// ***************************************
// **  settings dialog widget handling  **
// ***************************************
void Tlmb::setDlgShow()								// create and show settings dialog
{
//cout << "setDlgShow - reached " << endl;
	if (bSetDlgCreated == false) {
		bSetDlgCreated = true;
		setDlgW = new setDlg();															// settings dialog
		connect(this, SIGNAL(sigTlmbSetShow()), setDlgW, SLOT(dlgShow()));				// show settings dialog
		connect(setDlgW, SIGNAL(sigReReadSet(bool)), this, SLOT(setDlgClosed(bool)));	// settings dialog closed, re-read settings, might have been changed
	}
	wCtrl(3);
}

void Tlmb::setDlgClosed(bool bD)					// settings dialog was closed, delete it
{
//cout << "setDlgClosed - reached " << endl;
//	QCursor::setPos(pPos);
//	setUpdatesEnabled(true);
	setDlgW->deleteLater();
	sRootDirOld = sRootDir;
	iWShown = 0;
	bSetDlgCreated = false;
	if (bD == true) {
		settingsWrite();							// save selected comboboxes and current entries in all comboboxes, root dir already saved and updated by settings dialog
		settingsRead();
		if (sRootDir != sRootDirOld) {				// root path changed
			dirRootChgd();
			// try re-creating last session in new dir - hmm
		} else {
			// needed?
			cbIndicatorCreate();					// re-load and scale indicator icons
			cbIndicatorPlayingSet();				// clear and re-set playing indicator icon
			cbIndicatorPlAddToSet();				// clear and re-set add-to indicator icon
		}
	}
	wMainFocusSet();
}

// ****************************************
// **  songs table widget functions      **
// ****************************************
bool Tlmb::twSongsChk()								// check whether the selected comboboxes lead to songs to show
{
	if (iPlayCb > 0) {
		cbIndicatorPlayingSet();
		if (iPlayCb == 1) {								// playing from artist dir
			sSongsDir = pathGetAllowedDir(4);
			slPlaySongs = rwGetListSongs(pathGetAllowedDir(4));
		} else if (iPlayCb > 1 and iPlayCb < 5) {		// playing a playlist: root, music type or artist
			if (plPlayLoad() == true) {					// load StrM3u list
//cout << "twSongsChk - sPlayPlFull = " << sPlayPlFull.toStdString() << endl;                                               // *****  debug
				slPlaySongs = m3uSlSongsFromStrList(liM3uStrPlay);	// load stringlist, full path & name only from m3u/playlist struct list
			} else {
				slPlaySongs.clear();
				liM3uStrPlay.clear();
			}
		}

		if (slPlaySongs.count() == 1 and slPlaySongs.at(0).left(3) == "err") {	// error fetching playlist content
			slPlaySongs.clear();
			liM3uStrPlay.clear();
		}
		if (slPlaySongs.count() > 0) {
			twSongsList2StrList();
			return true;
		} else {
//cout << "tlmb: twSongsChk - return false: " << QString::number(slPlaySongs.count()).toStdString() << endl;                               // *****  debug  *****
			return false;
		}
	}
	return false;
}

void Tlmb::twSongsCellActivated(int iD, int iD2)	// cell in table widget activated (clicked) (row, col)
{
	iSongCurIdx = iD;
}

void Tlmb::twSongsCellDoubleClicked(int iD, int iD2)// cell in table widget double clicked (row, col) - play song
{
	iSongCurIdx = iD;
	uiTlmb.tlmb_tw_Songs->setCurrentCell(iD, 0, QItemSelectionModel::SelectCurrent);
//	playerPlay();
	playerStart();
}

void Tlmb::twSongsClear()							// clear table widget songs
{
	uiTlmb.tlmb_tw_Songs->clearContents();
	uiTlmb.tlmb_tw_Songs->setRowCount(0);
	sPlaySong = "";
	iSongCnt = 0;
//	iCurRowIx = -1;
//	iCurCol = -1;
}

void Tlmb::twSongsCon(bool bD)						// connect (true) / disconnect (false) songs table widget signals
{
	if (bD == true) {
		connect(uiTlmb.tlmb_tw_Songs->horizontalHeader(), SIGNAL(sectionResized(int, int, int)), this, SLOT(timerSaveHeaderWidthStart()));		// 5sec delay: check col headers w resized->save
//		connect(uiTlmb.tlmb_tw_Songs, SIGNAL(cellActivated(int, int)), this, SLOT(twSongsCellDoubleClicked(int, int)));							// song in table widget activated (ENTER pressed) - play it
		connect(uiTlmb.tlmb_tw_Songs, SIGNAL(cellDoubleClicked(int, int)), this, SLOT(twSongsCellDoubleClicked(int, int)));						// song in table widget double clicked - play it
		connect(uiTlmb.tlmb_tw_Songs, SIGNAL(currentCellChanged(int, int, int, int)), this, SLOT(twSongsCurrentCellChgd(int, int, int, int)));	// current cell changed, set (new?) row index
//		connect(uiTlmb.tlmb_tw_Songs, SIGNAL(customContextMenuRequested(const QPoint&)), this, SLOT(twSongsMenuCalled(const QPoint&)));
	} else {
		
	}
}

void Tlmb::twSongsCurrentCellChgd(int iD, int iD2, int iD3, int iD4)	// current cell changed - set (new?) row index
{
	iSongCurIdx = iD;
}

void Tlmb::twSongsHeaderSet()						// set table widget songs header
{
	uiTlmb.tlmb_tw_Songs->setColumnCount(2);
	uiTlmb.tlmb_tw_Songs->horizontalHeader()->setDefaultAlignment(Qt::AlignHCenter);
	uiTlmb.tlmb_tw_Songs->horizontalHeader()->setStretchLastSection(true);
	slD << tr("Song name     Sh+F9") << tr("File type");
	uiTlmb.tlmb_tw_Songs->setHorizontalHeaderLabels(slD);

	uiTlmb.tlmb_tw_Songs->setColumnWidth(0, iColWidth0);
	uiTlmb.tlmb_tw_Songs->setColumnWidth(1, iColWidth1);
	uiTlmb.tlmb_tw_Songs->setAlternatingRowColors(true);

}

void Tlmb::twSongsList2StrList()					// create list with song struct from list of playable songs (this list can come from rw.cpp or m3u.cpp - so do it here once)
{
	liSongStr.clear();
	if (iPlayCb == 1) {									// playing songs from directory
		if (slPlaySongs.count() > 0) {
			for (i = 0; i < slPlaySongs.count(); ++i) {		// iterate over file list, fill struct per file
				sD2 = slPlaySongs.at(i);
				strSongD.sPathName = sSongsDir + "/" + sD2;
				QFileInfo qfi(sD2);
					strSongD.sTyp = qfi.suffix();
					strSongD.sName = qfi.baseName(); 
				liSongStr.append(strSongD);
			}
		}
	} else if (iPlayCb > 1 and iPlayCb < 5) {			// playing songs from playlist
		if (liM3uStrPlay.count() > 0) {
			for (i = 0; i < liM3uStrPlay.count(); ++i) {		// iterate over file list, fill struct per file
				strM3uD = liM3uStrPlay.at(i);
				strSongD.sPathName = strM3uD.sSongPathName;
				QFileInfo qfi(strSongD.sPathName);
					strSongD.sTyp = qfi.suffix();
					strSongD.sName = qfi.baseName(); 
				liSongStr.append(strSongD);
			}
		}
	}
//	return liRwSongStr;
}

void Tlmb::twSongsMenuCalled(const QPoint& pPosD)	// right click in table widget songs - call context menu
{
	if (iAddToPlCb > 0) {
		pPos = pPosD;
		menuTwSongsRightClick->exec(QCursor::pos());
	}
}

void Tlmb::twSongsPop()								// populate songs widget
{
//cout << "tlmb: twSongsPop - reached " << endl;                               // *****  debug  *****
	twSongsCon(false);
	iSongCnt = liSongStr.count();
	uiTlmb.tlmb_tw_Songs->setRowCount(iSongCnt);
	for (i = 0; i < iSongCnt; ++i) {		// iterate over song list and populate widget
		strSongD = liSongStr.at(i);
		for (i2 = 0; i2 < 2; ++i2) {
			switch (i2) {
			    case 0:
				    twItemD = new QTableWidgetItem(strSongD.sName);
					twItemD->setTextAlignment(Qt::AlignLeft);
				    break;
			    case 1:
				    twItemD = new QTableWidgetItem(strSongD.sTyp);
					twItemD->setTextAlignment(Qt::AlignCenter);
				    break;
			}
			uiTlmb.tlmb_tw_Songs->setItem(i, i2, twItemD);
		}
	}
	uiTlmb.tlmb_tw_Songs->setCurrentCell(0, 0);
	twSongsCellActivated(0,0);
	uiTlmb.tlmb_tw_Songs->setFocus();
	twSongsCon(true);
}

void Tlmb::twSongsRightClickMenuCreate(int iD)		// right click menu for table widgets - iD = destination table widget
{
	menuTwSongsRightClick = new QMenu;
	menuTwSongsRightClick->clear();
	menuTwSongsRightClick->exec(QCursor::pos());
}

// ******************************************************************
// **  update widgets per level root, music type, artist and song  **
// ******************************************************************
bool Tlmb::updLevelArtist(bool bD)					// update artist level widgets, true= try enable, false=disable always: play combobox, play playlists combobox and add-to playlists combobox
{ 
	if (bD == true) {
		if (dirArtistChk() == true) {
			cbPlayArtistEnable(true);
			cbPlayArtistPop();
			cbPlayArtistSetCurrent(sPlayArtist);

//cout << "updLevelArtist: reached " << endl;                                               // *****  debug  *****		
			cbPlayPlArtistEnable(true);
			cbAddToPlArtistEnable(true);
			cbPlPopArtist();
			cbPlayPlArtistSetCurrent(sPlayPlArtist);
			if (iAddToPlCb == 7) {
				cbAddToPlSetCurrent();					// set current add-to playlist
			}
			return true;
		}
	}
	cbPlayArtistEnable(false);							// will also clear combobox and set sPlayMusicType to ""
	cbPlayPlArtistEnable(false);
	cbAddToPlArtistEnable(false);
	if (iAddToPlCb == 7) {
		cbAddToPlSetCurrent();							// set current add-to playlist
	}
	return false;
}

bool Tlmb::updLevelMusicType()						// update music type level widgets, play , play playlists and add-to playlists comboboxes
{
	if (dirMusicTypeChk() == true) {					// music type dir is valid: exists, readable and has music type dirs - sets slPlayMusicTypes when true
		cbPlayMusicTypeEnable(true);
		cbPlayMusicTypePop();
		cbPlayMusicTypeSetCurrent(sPlayMusicType);

		cbPlayPlMusicTypeEnable(true);
		cbAddToPlMusicTypeEnable(true);
		cbPlPopMusicType();
		cbPlayPlMusicTypeSetCurrent(sPlayPlMusicType);
		if (iAddToPlCb == 6) {
			cbAddToPlSetCurrent();						// set current add-to playlist
		}
		return true;
	} else {
		cbDisableAll();
		return false;
	}
}

bool Tlmb::updLevelRoot()							// update root level widgets, play and add-to playlists comboboxes
{
	if (dirRootChk() == true) {							// root dir is valid: exists and is readable
		cbPlayPlRootEnable(true);
		cbAddToPlRootEnable(true);
		cbPlPopRoot();
//cout << "updLevelRoot: cbAddToPlRoot->count() = " << QString::number(uiTlmb.tlmb_cb_root_playlists->count()).toStdString() << endl;                                               // *****  debug  *****
		cbPlayPlRootSetCurrent(sPlayPlRoot);
		if (iAddToPlCb == 5) {
			cbAddToPlSetCurrent();						// set current add-to playlist
		}
//cout << "updLevelRoot: reached " << endl;                                               // *****  debug  *****
		return true;
	} else {
		cbDisableAll();
		return false;
	}
}

bool Tlmb::updLevelSong(bool bDuLS)					// update songs level widgets, true= try enable, false=disable always: well, actually just the song table widget
{
	cbIndicatorPlayingSet();
	if (bDuLS == true) {
		if (twSongsChk() == true) {
			twSongsPop();
			return true;
		}
	}
//cout << "updLevelSong: before twSongsClear() " << endl;                                               // *****  debug  *****
	twSongsClear();
	return false;
}

// *******************************************
// **  widget handling    					**
// *******************************************
void Tlmb::wAddToPlSetDelete()						// hide set add-to playlists widget
{
	wPlSet->deleteLater();
	iWShown = 0;
	wMainFocusSet();
}

void Tlmb::wAddToPlSetNew(const QString& sD, bool bD)	// set new add-to playlist according to settings from the set add-to playlist widget - name, re-load combobox true=yes, false=no
{
	sAddToPl = sD;
	iAddToPlCb = iPlCbCalled;
	if (bD = true) {
		cbPlPopUpd(iAddToPlCb);
	}
	plAddToSetNew(sAddToPl);
}

void Tlmb::wAddToPlSetShow()						// show set add-to playlists widget
{
//cout << "plWSetShow: reached " << endl;                                               // *****  debug  *****
	if (iWShown == 0) {
		wPlSet = new SetPlayList();
		connect(this, SIGNAL(sigTlmbPlSetShow(const QString&, const QString&)), wPlSet, SLOT(splDlgShow(const QString&, const QString&)));
		connect(wPlSet, SIGNAL(sigSplTlmbM3uUpd(const QString&, bool)), this, SLOT(wAddToPlSetNew(const QString&, bool)));
		connect(wPlSet, SIGNAL(sigSplTlmbDelete()), this, SLOT(wAddToPlSetDelete()));
		iWShown = 5;
		plAddToSetPathName();
		emit sigTlmbPlSetShow(pathGetAllowedPl(iPlCbCalled), sAddToPlFull);	// **********  corresponding playlist comboboxes should be disabled when no root dir, music type/artist entry - no extra check here ??
	}
}


void Tlmb::wAhShowAbout()							// about clicked
{
	wCtrl(1);
}

void Tlmb::wAhShowHelp()							// help clicked
{
	wCtrl(2);
}

void Tlmb::wAhCreate()								// create about/help/error dialog widget
{
	if (iWShown == 0) {
		ahw = new aboutHelp();
		connect(this, SIGNAL(sigTlmbAhwShow(int)), ahw, SLOT(dlgShow(int)));
		connect(ahw, SIGNAL(sigDelAhw()), this, SLOT(wAhDel()));
	}
}

void Tlmb::wAhDel()									// ahw widget was been hidden - delete it
{
	ahw->deleteLater();
	iWShown = 0;
	wMainFocusSet();
	if (bQuit) {
		quitPrep();
	}
}


void Tlmb::wCtrl(int iWToShow)						// show popup widget, if no other is shown: 0=none, 1=about, 2=help, 3=settings dialog, 4=playing Pl edit, 5=add to Pl set, 6=add to Pl edit
{
	if (iWShown == 0) {								// only proceed, if no other widget is shown
		switch (iWToShow) {
			case 1:										// show about widget
				wAhCreate();
				iWShown = 1;
				emit sigTlmbAhwShow(iWShown);
				break;
			case 2:										// show help widget
				wAhCreate();
				iWShown = 2;
				emit sigTlmbAhwShow(iWShown);
				break;
			case 3:										// show settings dialog
				iWShown = 3;
//				setCurType(0);
//				setUpdatesEnabled(false);
//				qpD = QCursor::pos();
				emit sigTlmbSetShow();
				break;
		}
	}
}

void Tlmb::wMainFocusSet()							// set focus (back) to main widget or if possible to song tablewidget
{
	if (uiTlmb.tlmb_tw_Songs->rowCount() > 0) {
		uiTlmb.tlmb_tw_Songs->setFocus();
	} else {
		this->setFocus();
	}
}


void Tlmb::wPlEditApply(const QList<StrM3u>& liM3uStr)	// apply clicked on the edit playlist widget - apply any changes
{
	if (plWrite(sPlPath + "/" + sPl, liM3uStr) == true) {
		mbTxt_WPlEditSuccess();
		if (iPlayCb == iPlCbEditShown) {				// playing playlist was changed - re-load
			if (twSongsChk() == true) {
				twSongsPop();
			}
			if (iAddToPlCb == iPlayCb + 3) {				// corner case, playing add-to playlist
				liM3uStrAddTo = liM3uStr;
			}
		}
		if (iAddToPlCb == iPlCbEditShown) {				// add-to playlist was changed - update
			liM3uStrAddTo = liM3uStr;
			if (iAddToPlCb == iPlayCb + 3) {				// corner case, playing add-to playlist
				if (twSongsChk() == true) {
				twSongsPop();
				}
			}
		}
	} else {
		mbTxt_WPlEditFailed();
	}
	wPlEditDelete();
}

void Tlmb::wPlEditDelete()							// delete edit playlist widget
{
	wPlEdit->deleteLater();
	iWShown = 0;
	wMainFocusSet();
}

void Tlmb::wPlEditShow()							// show edit playlist widget
{
	if (iWShown == 0) {
		iPlCbEditShown = iPlCbCalled;
		sPlPath = pathGetAllowedPl(iPlCbEditShown);

		switch (iPlCbEditShown) {
			case 2:
				sPl = uiTlmb.tlmb_cb_root_playlists->currentText();
				bPlAddToPlay = false;
				break;
			case 3:
				sPl = uiTlmb.tlmb_cb_musicType_playlists->currentText();
				bPlAddToPlay = false;
				break;
			case 4:
				sPl = uiTlmb.tlmb_cb_artist_playlists->currentText();
				bPlAddToPlay = false;
				break;
			case 5:
				sPl = uiTlmb.tlmb_cb_plAddTo_root->currentText();
				bPlAddToPlay = true;
				break;
			case 6:
				sPl = uiTlmb.tlmb_cb_plAddTo_musicType->currentText();
				bPlAddToPlay = true;
				break;
			case 7:
				sPl = uiTlmb.tlmb_cb_plAddTo_artist->currentText();
				bPlAddToPlay = true;
				break;
		}

		if (sPl == "") {								// no playlist
			return;
		}

		if (plLoad(bPlAddToPlay, sPlPath + "/" + sPl) == true) {
			if (liM3uStr.count() > 0) {					// do not show empty playlist - struct list does not have first line #EXTM3U entry, 1 line per double line m3u entry
				wPlEdit = new EditPlayList();
				connect(this, SIGNAL(sigTlmbPlEditShow(const QString&, const QString&, const QList<StrM3u>&)), wPlEdit, SLOT(eplDlgShow(const QString&,const QString&, const QList<StrM3u>&)));
				connect(wPlEdit, SIGNAL(sigEplTlmbM3uUpd(const QList<StrM3u>&)), this, SLOT(wPlEditApply(const QList<StrM3u>&)));
				connect(wPlEdit, SIGNAL(sigEplTlmbDelete()), this, SLOT(wPlEditDelete()));
				iWShown = 6;
				emit sigTlmbPlEditShow(sPlPath, sPlPath + "/" + sPl, liM3uStr);	// allowed path, m3u full path name + m3u struct list
			} else {
				mbTxt_WPlEditEmpty();
			}
		}
	}
}

// ********************************************************
// **  timers											 **
// ********************************************************
void Tlmb::timersCreate()							// create all timers
{
	timerSaveHeaderWidth = new QTimer(this);										// 5 sec timer to limit checks if the table column headers where resized
	timerSaveHeaderWidth->setSingleShot(true);
	connect(timerSaveHeaderWidth, SIGNAL(timeout()), this, SLOT(settingsWrite()));	// save new column width of history table widget

	timerStopPlayingDisable = new QTimer(this);										// 5 sec timer to limit checks if the table column headers where resized
	timerStopPlayingDisable->setSingleShot(true);
	connect(timerStopPlayingDisable, SIGNAL(timeout()), this, SLOT(playerDisableStop()));	// disable stop playing next song when playing to end of song list or loop through it
}

void Tlmb::timerSaveHeaderWidthStart()				// start timer to save header width of all qtablewidgets
{
	if (timerSaveHeaderWidth->isActive() == false) {
		timerSaveHeaderWidth->start(5000);
	}
}

void Tlmb::timerStopPlayingDisableStart()			// start timer to disable stop playing next song when playing to end of song list or loop through it
{
	if (timerStopPlayingDisable->isActive() == false) {
		timerStopPlayingDisable->start(5000);
	}
}
