/*********************************************************************
 *  SPDX-License-Identifier: MIT                                     *
 *  Copyright (C) 2022-2023 by Helge Kortendieck  alan@kasmithee.de  *
 *                                                                   *
 *  This file is part of tlmb.                                       *
 *********************************************************************/
#include "playlistset.h"

SetPlayList::SetPlayList(QDialog* parent)
	: QDialog(parent)
{
	uiSpl.setupUi(&wSpl);
	uiSpl.pb_New->setIcon(QIcon("/usr/share/tlmb/icons/scalable/new.svg"));
	uiSpl.pb_Rename->setIcon(QIcon("/usr/share/tlmb/icons/scalable/rename.svg"));
	uiSpl.pb_SaveAs->setIcon(QIcon("/usr/share/tlmb/icons/scalable/save-as.svg"));
	connect(uiSpl.pb_Set, SIGNAL(clicked()), this, SLOT(splDlgSetNew()));
	connect(uiSpl.pb_Cancel, SIGNAL(clicked()), this, SLOT(splDlgClose()));
	connect(uiSpl.pb_New, SIGNAL(clicked()), this, SLOT(splFileNew()));
	connect(uiSpl.pb_Rename, SIGNAL(clicked()), this, SLOT(splFileRename()));
	connect(uiSpl.pb_SaveAs, SIGNAL(clicked()), this, SLOT(splFileSaveAs()));
	uiSpl.pb_Cancel->setFocus();
}

SetPlayList::~SetPlayList()
{
}


//  *******************************************
//  **  check for/create playlist directory  **
//  *******************************************
bool SetPlayList::splPlDirChk()					// check if .../playlists directory exists - ask to create and eventually do if not
{
	QFileInfo fi(sSplPathAllowed);
	if (fi.exists() == true) {						// playlist dir exists
		return true;
	} else {
		return false;
	}
}

bool SetPlayList::splPlDirCreate()				// ask to create .../playlists directory and eventually do
{
	iSplRet = splQmbQ(41);						// no playlists dir - create playlists directory?
	if (iSplRet == QMessageBox::Ok) {
		QDir qDir;
		bSplD = qDir.mkdir(sSplPathAllowed);
		if (bSplD == true) {
			return true;
		} else {
			splQmb(42);							// failed to create the .../playlists directory
			return false;
		}
	} else if (iSplRet == QMessageBox::Cancel) {
		return false;
	} else {
		return false;
	}
}


//  *********************************
//  **  playlist dialog functions  **
//  *********************************
void SetPlayList::splDlgShow(const QString& sSplD, const QString& sSplD2)	// show song list, current dir or favourites songs - pathAllowed, FilePathNameCalled
{
	sSplPathAllowed = sSplD;
	sSplFilePathNameCalled = sSplD2;
	sSplFilePathName = sSplFilePathNameCalled;
//cout << "SetPlayList - sSplFilePathName = " << sSplFilePathNameCalled.toStdString() << endl;                            // *****  debug  *****
	splDlgPopPathName();

	wSpl.show();
}

void SetPlayList::splDlgSetNew()				// set new m3u
{
	if (bSplCreateNew == true) {
		if (splFileNewCreate() == 0) {				// could not create new playlist
			bSplCreateNew = false;
			splDlgPopPathName();
			splFileCurChngd();
			return;
		}
	}
//cout << "splDlgSetNew - sSplFileName = " << sSplFileName.toStdString() << endl;                                               // *****  debug  *****
	emit sigSplTlmbM3uUpd(sSplFileName + ".m3u", true);
	splDlgClose();
}

void SetPlayList::splDlgClose()
{
	emit sigSplTlmbDelete();
}

void SetPlayList::splDlgPopPathName()
{
	QFileInfo fi(sSplFilePathName);
		sSplFilePath = fi.absolutePath();
		if (sSplPathAllowed == sSplFilePath) {
			sSplFileName = fi.baseName();
		} else {									// current playlist is in another directory - cannot be set in this level, unset name
			sSplFileName = "";
		}
	uiSpl.l_FilePath->setText(sSplPathAllowed);
	uiSpl.l_FileName->setText(sSplFileName);
}


//  *******************************
//  **  playlist file functions  **
//  *******************************
void SetPlayList::splFileCurChngd()				// current m3u changed
{
	if (sSplFilePathName != sSplFilePathNameCalled) {
		uiSpl.pb_Set->setEnabled(true);
	} else {
		uiSpl.pb_Set->setEnabled(false);
	}
}

void SetPlayList::splFileNew()					// create new playlist to add files to
{
	if (splPlDirChk() == false) {					// do not have .../playlists directory
		if (splPlDirCreate() == false) {				// could not create ../playlists directory
			return;											// bail out - message from fail already shown while trying to create
		}
	}

	bSplCreateNew = false;
	QFileDialog fd(this, tr("Create new, empty playlist"), sSplPathAllowed, "*.m3u");
		fd.setViewMode(QFileDialog::List);
		fd.setFileMode(QFileDialog::AnyFile);
		fd.setAcceptMode(QFileDialog::AcceptSave);
		fd.setDefaultSuffix("m3u");
	
	if (fd.exec()) {
		slSplD = fd.selectedFiles();
	}
	if (slSplD.count() > 0) {						// dialog not canceled
		sSplFileFromDlg = slSplD.at(0);
		if (sSplFileFromDlg != "") {
			QFileInfo fi(sSplFileFromDlg);
			if (fi.absolutePath() != sSplPathAllowed) {
				splQmb(2);
			} else {
				bSplCreateNew = true;
				sSplFilePathName = sSplFileFromDlg;
//cout << "splFileNew - sSplFilePathName = " << sSplFilePathName.toStdString() << endl;                                               // *****  debug  *****
				splDlgPopPathName();
				splFileCurChngd();
			}
		} else {
			splQmb(1);								// empty/ no file name - should never happen
		}
	}
}

int SetPlayList::splFileNewCreate()				// physically create new playlist to add files to - 0=fail, 1=success
{
	if (m3uFileCreate(sSplFilePathName) == "0") {
		splQmb(11);									// new playlist could not be created
		return 0;
	} else {
		return 1;
	}
}

void SetPlayList::splFileRename()				// rename current playlist
{
	QFileDialog fd(this, tr("rename playlist"), sSplPathAllowed, "*.m3u");
		fd.setViewMode(QFileDialog::Detail);
		fd.setFileMode(QFileDialog::AnyFile);
		fd.selectUrl(QUrl::fromLocalFile(sSplPathAllowed + "/" + sSplFileName));
		fd.setDefaultSuffix("m3u");

	if (fd.exec() == 1) {
		slSplD = fd.selectedFiles();
	} else {
		return;
	}
	if (slSplD.count() > 0) {
		sSplFileFromDlg = slSplD.at(0);
	} else {
		return;
	}

	QFileInfo fi(sSplFileFromDlg);
	if (fi.exists() == true) {
		splQmb(21);									// invalid playlist name, already exists
	} else {
		QMessageBox msgBox;
//			msgBox.setWindowTitle( tr( "Rename ?" ) );
			msgBox.setIcon( QMessageBox::Question );
			msgBox.setText( tr( "\nSure to rename m3u ?\n" ) );
			msgBox.addButton( tr( "Yes, &Rename" ), QMessageBox::AcceptRole );
			msgBox.addButton( tr( "No, &Cancel" ), QMessageBox::RejectRole );
		if (msgBox.exec() == 0) {
			if (m3uFileRename(sSplFilePathName, sSplFileFromDlg) == false) {
				splQmb(22);								// error while renaming
			} else {
				sSplFilePathName = sSplFileFromDlg;
				splDlgPopPathName();
				splQmb(23);								// playlist successfully renamed
				splDlgSetNew();
			}
		}
	}
}

void SetPlayList::splFileSaveAs()				// save current playlist under different name and start using it as such
{
	QFileDialog fd(this, tr("Save current playlist under new name"), sSplFilePath, "*.m3u");
		fd.setViewMode(QFileDialog::Detail);
		fd.setFileMode(QFileDialog::AnyFile);
		fd.setDefaultSuffix("m3u");
	
	if (fd.exec()) {
		slSplD = fd.selectedFiles();
	}
	sSplFileFromDlg = slSplD.at(0);

	if (sSplFileFromDlg != "") {
		if (m3uFileSaveAs(sSplFilePathName, sSplFileFromDlg) == true) {
			sSplFilePathName = sSplFileFromDlg;
			splDlgPopPathName();
			splQmb(31);								// successfully saved under new name
			splDlgSetNew();
		} else {
			splQmb(32);								// error saving under new name
		}
	}
}

void SetPlayList::splQmb(int iSplD)				// QMessageBox
{
	QMessageBox msgBox;
		switch (iSplD) {
			case 1:									// splFileNew - empty/no m3u playlist name
				msgBox.setText("Error - Empty/no m3u playlist name - should never happen!?");
				msgBox.setInformativeText("Sigh, might want to try again?");
				break;
			case 2:									// splFileNew - wrong path, changing path is not allowed
				msgBox.setText("Error - Wrong path!\n It is not allowed to change the path.\n Cannot prevent to do so in the file dialog!\n Please only enter the file name of the new playlist.");
				msgBox.setInformativeText("Sigh, might want to try again?");
				break;
			case 11:								// splFileNewCreate
				msgBox.setText("Error - It seems, the new playlist could not be create.");
				msgBox.setInformativeText("May be the file exists or a write permission problem?");
				break;
			case 21:								// splFileNewRename - name invalid/already exists
				msgBox.setText("Error - Invalid playlist name.");
				msgBox.setInformativeText("A file with this name already exists!");
				break;
			case 22:								// splFileNewRename - error while renaming
				msgBox.setText("Error - while renaming playlist.");
				msgBox.setInformativeText("An error occurred while trying to rename the current playlist!");
				break;
			case 23:								// splFileNewRename - success
				msgBox.setText("Success - renaming playlist.");
				msgBox.setInformativeText("The playlist has been renamed!");
				break;
			case 31:								// splFileSaveAs - success
				msgBox.setText("Success - saving playlist under new name.");
				msgBox.setInformativeText("The playlist has been saved under the new name!");
				break;
			case 32:								// splFileSaveAs - error
				msgBox.setText("Error - It seems, the playlist could not be saved under the new name.");
				msgBox.setInformativeText("May be the file exists or a write permission problem?");
				break;
			case 42:								// splDirChk - error - could not create playlists dir
				msgBox.setText("Error - It seems, the playlists directory could not be created.");
				msgBox.setInformativeText("As the directory should not already exist, it might be due to a write permission problem?");
				break;
		}
		msgBox.exec();
}

int SetPlayList::splQmbQ(int iSplD)				// QMessageBox question
{
	QMessageBox msgBox;
	switch (iSplD) {
		case 41:								// splDirChk - create .../playlists dir ?
			msgBox.setText("Question - playlists directory does not exist - create?");
			msgBox.setInformativeText("The .../playlists directory holding the *.m3u playlist files does not (yet) exist in this directory:\n"
									  "Shall it be created?\n"
									  "Without it, no playlist can be created in this directory!");
			msgBox.setStandardButtons(QMessageBox::Ok | QMessageBox::Cancel);
			msgBox.setDefaultButton(QMessageBox::Ok);
			break;
	}
	iSplRet = msgBox.exec();
	return iSplRet;
}
