/*********************************************************************
 *  SPDX-License-Identifier: MIT									 *
 *  Copyright (C) 2006-2024 by Helge Kortendieck  alan@kasmithee.de  *
 *                                                                   *
 *  This file is part of qrsyncmgr.                                  *
 *********************************************************************/
#ifndef _RWDEFS_H_
#define _RWDEFS_H_

#include <QDir>
#include <QFile>
#include <QFileInfo>
#include <QObject>
#include <QString>
#include <QStringList>
#include <QTextStream>

#include "structs.h"

class rwdefs : public QObject
{
  Q_OBJECT

public:
  rwdefs(QObject *parent = 0);
    ~rwdefs();

private:
  bool bWriteErr;           // error while writing rsync definition

  int iCase;                // read/write case, 1=add, 2=upd, 3=del db entry (record)
  int iD;                   // dummy int, use locally
  int iD2;
  int iErrDirChk;           // error status while checking directory with rsync definitions: exists 0/1 + readable 0/2 + writeable 0/4 + has defs 0/8 = 0,1,3,7,11,15
  int iRead;                // read case, 0 = one line entries, 1 = add description line, 2 = add command line

  Rsync rsync;              // dummy item of a single rsync definition
  Rsync rsyncD;             // dummy item of a single rsync definition

  QString sD;               // the usual short living variable
  QString sD2;              // the usual short living variable
  QString sDir;             // directory with rsync definition, incl. '/' at end !!
  QString sDirPath;         // full path and name of dir with rsync definitions
  QString sErr;             // error text
  QString sErrTit;          // error dialog titel
  QString sFullName;        // full name of rsync definition file incl. full path
  QString sIn;              // line read from file
  QString sName;            // name of rsync, excl. path and '.rd' ending
  QString sOutTxt;          // text to add to master output widget
  QString sRI;              // file name of rsync to import

  QString sLastRun;         // last run date (string)
  QString sDesc;            // rsync description
  QString sSrcMp;           // source dir mount command, excl. 'mount'
  QString sDestMp;          // destination dir mount point
  QString sRsyncs;          // rsync command(s), one per linedestination dir mount point
  QString sCmd;             // rsync command to execute now

  QStringList slD;          // dummy qstringlist
  QStringList slRsyncs;     // qstringlist with rsync definition file names of current dir

private slots:
  void rsyncAdd(const Rsync& rsyncD);     // add this rsync definition
  void rsyncClear();                      // clear all rsync definition variables
  void rsyncDel(const QString& sD);       // delete this rsync definition
  void rsyncFullName();                   // set the full name of the rsync definition file incl. full path
  void rsyncGet(const QString& sD);       // get the rsync definition details (fields)
  void rsyncUpd(const Rsync& rsyncD);     // update this rsync definition
  void rsyncWrite();                      // write file with newly added or updated rsync definition

  void dirCheck(const QString& sD);       // check n create ~/.kde/share/apps/rsynkmgr if required
  void readRsyncs();        // read defined rsyncs (names) = dirs in ~/.kde/share/apps/rsynkmgr/
  void rsyncWriteOk();      // rsync write operation successfully (add, update or deleted), show in output widget

  signals:
    void sigDirChkd(const QString& sErrTit, const QString& sErr, int iErrDirChk);
    void sigErr(const QString& sErrTit, const QString& sErr); // and error occured in sql3 qobject (this)
    void sigRsyncsRead(const QStringList& slRsyncs);          // read rsync definition file names (rsync names) found in directory
    void sigUpdDetails(const Rsync& rsync);       // rsync details read, go grab them and show
    void sigUpdOutput(const QString& sD, int iD); // update textbox with output from rwdefs qobject
    void sigToClient();                           // switch to client mode, cannot start master mode
};

#endif // _RWDEFS_H_
