/*********************************************************************
 *  SPDX-License-Identifier: MIT									 *
 *  Copyright (C) 2006-2024 by Helge Kortendieck  alan@kasmithee.de  *
 *                                                                   *
 *  This file is part of qrsyncmgr.                                  *
 *********************************************************************/
#include "rwdefs.h"

rwdefs::rwdefs(QObject *parent) : QObject(parent)
{
//  sDirPath = "/tmp";
}

rwdefs::~rwdefs()
{
}


void rwdefs::dirCheck(const QString& sD)  // check given dir with rsyncs, at first start use users home dir (set during read settings)
{
  sDir = sD;
  if (sDir.right(1) != "/") {
    sDir = sDir + "/";
  }
  iErrDirChk = 0;
  QDir d( sDir );                              // returns dir excl. '/' at the end
    if (d.exists() == true) {                  // dir exists +0/1
      iErrDirChk = 1;
      if (d.isReadable() == true) {            // dir readable +0/2
        iErrDirChk = iErrDirChk + 2;
        QFileInfo fi( sDir );                  // dir writable +0/4
        if (fi.isWritable() == true) {
          iErrDirChk = iErrDirChk + 4;
        }
        readRsyncs();                          // read rsync definitions in dir
        if (slRsyncs.count() > 0) {            // found rsync definitions +0/8
          iErrDirChk = iErrDirChk + 8;
        }
      }
    }

  sErrTit = "qrsyncmgr - Error while checking dir with rsync defs !";
  switch (iErrDirChk) {
    case 0:                                                         // dir does not exist
      sErr = tr( "Directory\n%1\ndoes not exist !\n \n"
                 "Please change the directory, so you can read/create rsync definitions.\n"
                 "(At the top left of the up-coming program window.)" ).arg( sDir );
      break;
    case 1:                                                         // dir exists, but not read-/writable, no defs? (hey, not readable !)
      sErr = tr( "Directory\n%1\nis not readable !\n"
                 "Please change the directory or the directories permission, so you can read any rsync definitions." ).arg( sDir );
      break;
    case 3:                                                         // dir exists, readable, not writable, no defs
      sErr = tr( "Directory\n%1\nis not writable and does not contain any rsync definitinons !\n"
                 "Please change the directory or the directories permission, so you can create any rsync definitions." ).arg( sDir );
      break;
    case 7:                                                         // dir exists, read/writeable, no defs
      sErr = tr( "Directory\n%1\ndoes not contain any rsync definition !\n"
                 "Nothing to do.\nContinue and define a rsync operation or switch to a directory which contains rsync definitions." ).arg( sDir );
      break;
    case 11:                                                        // dir exists, readable, has defs, but not writable
      sErr = tr( "Directory\n%1\nhas rsync definitions but is not writable !\n"
                 "You can run the defined rsync(s) but you can not alter it/them.\n"
                 "You must change the directories permission first, before you can add/change/delete any rsync definitions." ).arg( sDir );
      break;
    case 15:                                                        // dir exists, read/writeable, has defs
      sErrTit = tr( "qrsyncmgr - All clear !" );
      sErr = tr( "Directory\n%1\nis read- n writable and does contain rsync definitions !" ).arg( sDir );
      break;
  }
  emit sigDirChkd( sErrTit, sErr, iErrDirChk );
}


void rwdefs::readRsyncs()                    // read defined rsyncs (names) from directory
{
  slRsyncs.clear();
  QDir d(sDir);
    d.setFilter(QDir::Files);
    slD.clear();
    slD << "*.rd";
    d.setNameFilters(slD);
    d.setSorting(QDir::Name);
  slRsyncs = d.entryList();
  slRsyncs.replaceInStrings(".rd", "");       // get rid of the *.rd file type

  emit sigRsyncsRead(slRsyncs);
}

void rwdefs::rsyncFullName()                // create rsync definition full file name incl. path
{
  sFullName = sDir + sName + ".rd";
}

void rwdefs::rsyncAdd(const Rsync& rsyncD)  // add new rsync definition
{
  rsync = rsyncD;
  sName = rsync.name;
  iCase = 1;                                    // read/write case, 1=add, 2=upd, 3=del definition file

  rsyncFullName();
  QFileInfo fi(sFullName);
  if (fi.exists()) {                           // dir exists, rsync already defined, cannot create
    sErrTit = tr("Error - adding rsync definition");
    sErr = tr("A rsync with this name does already exist.\n"
              "The name has to be unique!\n"
              "Use the 'Upd' button to update an existing definiton.\n");
    emit sigErr(sErrTit, sErr);
  } else {                                      // dir does not exist, can add rsync definition
    sErrTit = tr("Error - writing rsync definition");
    sErr = tr("Cannot create the file\n%1\n"
              "This is strange. Probably not a permission problem. Out of disk space?\n"
              "Error returned:\n").arg(sFullName);
    rsyncWrite();
  }
}


void rwdefs::rsyncUpd(const Rsync& rsyncD)  // update a rsync definition
{
  rsync = rsyncD;
  sName = rsync.name;
  iCase = 2;

  rsyncFullName();
  QFileInfo fi(sFullName);
  if(!fi.exists()) {                          // dir does not exists, cannot update
    sErrTit = tr("Error - updating rsync definition");
    sErr = tr("There is no rsync with this name defined!\n"
              "Use the 'Add' button to create a rsync for the first time.\n");
    emit sigErr(sErrTit, sErr);
  } else {                                    // dir does exist, update rsync definition
    sD = sDir + sName + ".tmp";               // full name of *.tmp file
    QDir fileold(sFullName);                  // rename current *.rd file to *.tmp
    if (fileold.rename(sFullName , sD)) {
      sErrTit = tr("Error - writing *.rd file");
      sErr = tr("Could not write the file with the updated rsync definitions\n%1\n"
                "That should never happen. You may have disk space problems!\n"
                "Actual changes will be lost, when you close the program.\n"
                "May be you can fix the issue with the program still running and try saving again?\n"
                "If not, the previous definition is still available as:\n%2\n"
                "You can re-name it back to\n%3\nwith a file manager (mc, Thunar, Dolphin, Nautilus).\n"
                "Error returned:\n").arg(sFullName).arg(sD).arg(sFullName);
      rsyncWrite();
      if (bWriteErr == false) {
        QFile file(sD);                       // delete temp file *.tmp
          if (file.remove()) {
            rsyncWriteOk();                     // updating the rsync definition seems to be successfully
          } else {
            sErrTit = tr("Error - removing *.tmp file");
            sErr = tr("Could not remove the temporary definition file\n%1\n"
                      "It holds the rsync definition before last update was called.\n"
                      "Error returned: %2\n \n"
                      "Aborting (hit OK)").arg(sD).arg(file.errorString());
            emit sigErr(sErrTit, sErr);
          }
      } else {
        sErrTit = tr("Error - renaming *.rd file");
        sErr = tr("Cannot rename file:\n%1\n"
                  "to: \n%2\n"
                  "This could indicate a file permission problem.\n \n"
                  "Aborting (hit OK)").arg(sFullName).arg(sD);
        emit sigErr(sErrTit, sErr);
      }
    }
  }
}

void rwdefs::rsyncWrite()                     // write file with newly added or updated rsync definition
{
  bWriteErr = false;
  QFile file(sFullName);
  if (file.open(QIODevice::WriteOnly)) {
    QTextStream stream(&file);
      stream << "LastRun=" << rsync.lastRun << "\n";
      stream << "Desc=" << rsync.desc << "\n";
      stream << "SrcMp=" << rsync.srcMp << "\n";
      stream << "DestMp=" << rsync.destMp << "\n";
      stream << "Cmd=" << rsync.cmd << "\n";
    file.close();
    if (iCase == 1) {                          // if updating (iCase = 2 ), try to remove tmp file first before send OK signal
      rsyncWriteOk();                           // writing the rsync definition seems to be successfully
    }
  } else {
    bWriteErr = true;
    emit sigErr(sErrTit, sErr + tr("%1\n \nAborting (hit OK)").arg(file.errorString()));
  }
}

void rwdefs::rsyncDel(const QString& sD)  // delete rsync definition
{
  sName = sD;
  iCase = 3;
  rsyncFullName();
  QFile file(sFullName);                      // delete *.rd
  if (file.remove()) {
    rsyncWriteOk();                             // deleting the rsync definition seems to be successfully
  } else {
    sErrTit = tr("Error - deleting *.rd file");
    sErr = tr("Could not delete the rsync definition file\n%1\n"
              "This could indicate a file permission problem.\n"
              "Error returned: \n%2\n \n"
              "Aborting (hit OK)").arg(sFullName).arg(file.errorString());
    emit sigErr(sErrTit, sErr);
  }
}


void rwdefs::rsyncGet(const QString& sD)  // get rsync definition details (fields)
{
  sName = sD;
  iRead = 0;
  rsyncClear();
  rsyncFullName();                              // translate the rsync def name into full path and name of definition
  QFile file(sFullName);
  if (file.open(QIODevice::ReadOnly)) {
    QTextStream in(&file);
    while (!in.atEnd()) {                       // read the file (textstream) line by line
      sIn = in.readLine();
      if (sIn != "") {                         // ignore empty lines
        if (sIn.left(8) == "LastRun=") {
          rsync.lastRun = sIn.remove(0, 8);
          iRead = 0;
        } else if (sIn.left(5) == "Desc=") {
          sIn.remove(0, 5);
          iRead = 1;
        } else if (sIn.left(6) == "SrcMp=") {
          rsync.srcMp = sIn.remove(0, 6);
          iRead = 0;
        } else if (sIn.left(7) == "DestMp=") {
          rsync.destMp = sIn.remove(0, 7);
          iRead = 0;
        } else if (sIn.left(4) == "Cmd=") {
          sIn.remove(0, 4);
          iRead = 2;                            // read case, 0 = one line entries, 1 = add description line, 2 = add command line
        }
        switch (iRead) {
          case 1:                                     // add further description lines
            rsync.desc = rsync.desc + sIn + "\n";
            break;
          case 2:                                     // add further command lines
            rsync.cmd = rsync.cmd + sIn + "\n";
            break;
        }
      }
    }
    file.close();                               // close file at EOF
    QFileInfo fi(sFullName);
      sD2 = fi.fileName();
      iD = sD2.length();                        // chop does not seem to work?
      rsync.name = sD2.left(iD - 3);
    emit sigUpdDetails(rsync);
  } else {                                      // could not open rsync conf file to read details
    sErrTit = tr("Error - opening file");
    sErr = tr("Could not read the details of this rsync definition.\n"
              "Could not open file: \n%1\n"
              "This could indicate a file (directory) permission problem or an\n"
              "unclean earlier deletion = no definition file in this rsync directory.\n \n"
              "Aborting (hit OK)").arg(sFullName);
    emit sigErr(sErrTit, sErr);
  }
}

void rwdefs::rsyncClear()                   // clear all rsync definition variables
{
  rsync.lastRun = "";
  rsync.desc = "";
  rsync.srcMp = "";
  rsync.destMp = "";
  rsync.cmd = "";
}

void rwdefs::rsyncWriteOk()                 // rsync operation successfully (add, update or delete)
{
  switch (iCase) {
    case 1:                                     // rsync definition added successfully
      sD = tr("added");
      break;
    case 2:                                     // rsync definition updated successfully
      sD = tr("updated");
      break;
    case 3:                                     // rsync definition deleted successfully
      sD = tr("deleted");
      break;
  }
  sOutTxt = tr("Rsync definition '%1' %2.").arg(sName).arg(sD);
  readRsyncs();                                 // re-read the list of defined rsyncs
  emit sigUpdOutput(sOutTxt, 1);
}


//#include "rwdefs.moc"
