/*********************************************************************
 *  SPDX-License-Identifier: MIT									 *
 *  Copyright (C) 2006-2024 by Helge Kortendieck  alan@kasmithee.de  *
 *                                                                   *
 *  This file is part of qrsyncmgr.                                  *
 *********************************************************************/
#ifndef _RSYNC_H_
#define _RSYNC_H_

#include <QByteArray>
#include <QDateTime>
#include <QFileInfo>
#include <QObject>
#include <QProcess>
#include <QProcessEnvironment>
#include <QRegularExpression>
#include <QString>
#include <QTimer>

#include <iostream>
using namespace std;

class rsync : public QObject
{
  Q_OBJECT

public:
	rsync(QObject *parent = 0);
		~rsync();

private:
	bool bErr;				// flag whether we have error while setting up/down services
	bool bRStop;			// flag whether to stop a rsync
	bool bStarted;			// need to disable catch of qprocess output once logged in, otherwise conflict with path/file names

	int iArgs;				// counter for arguments, we need at least 3!
	int iD;
	int iD2;
	int iChk;				// check for rsync and askpass: 0=none installed, 1=rsync only, 2=askpass only, 3=rsync and askpass
	int iErr;				// internal error code, set during read of stdout and stderr
	int iErr32;				// rsync error (1), failed to set times on a directory, directory permission problem
	int iErr64;				// rsync error (13), failed to mkstemp on a file, file permission problem
	int iHr;				// rsync duration hr, min, sec
	int iMin;
	int iRsys;				// number of rsyncs defined in this rsync definition
	int iRsy;				// actual rsync of the list of defined rsyncs in this rsync definition
	int iSec;

	QByteArray aOutput;		// stdout from mntp, mounting qprocess

	QProcess *ask;			// qprocess to check for ssh-askpass
	QProcess *rsy;			// rsync qprocess

//	QString sAskPass;		// name of askpass program, x11-ssh-askpass or ssh-askpass ************** needed?
	QString sAskPassPath;	// path to askpass, if not in /usr/bin, will be added as environment variable to qprocess
	QString sCmd;			// rsync command for qprocess
	QString sD;
	QString sErrT;			// rsync error message title
	QString sErr;			// rsync error message
	QString sIn;			// input line from rsync qprocess output
	QString sLastRun;		// last run date, formatted if rsync was successful
	QString sLine;			// rsync command line
	QString sMsg;			// rsync output
	QString sRsys;			// rsync command(s) as defined in current rsync definition

	QStringList slArgs;		// qprocess argument(s)
	QStringList slRsys;		// list with rsyncs of current rsync definition

	QTime elap;				// timer for rsync duration

private slots:
	void rsyChkProgs();		// check for rsync and ssh-askpass
	void rsyPrep(const QString& sD);	// prepare rsync, QString can have many lines (rsync definitions)

	void procPrep();		// prepare rsync run
	void procStart();		// start rsync qprocess
	void procErr();			// error, during rsync run, send error message
	void procExit(int iD, QProcess::ExitStatus iD2);	// rsync process exited
	void procPwErr();		// wrong ssh password or user canceled password input
	void procStop();		// stop rsync process
	void procReadStdErr();	// read error output from rsync process
	void procReadStdOut();	// read output from rsync process

	signals:
		void sigChkEnd(int);	// checking for rsync and askpass has ended. Result: 0=none installed, 1=rsync only, 2=askpass only, 3=rsync+askpass
		void sigErr(const QString& sErrT, const QString& sErr);		// rsync failed
		void sigOk(const QString& sD, int iD, const QString& sErr);	// rsync appeared to be ended successfully: last run date, error, warning string
		void sigStopped();												// rsync has been stopped by user request, rsync process has ended
		void sigUpdOutput(const QString& sD, int iD);					// update output widget
};

#endif // _RSYNC_H_
