/*********************************************************************
 *  SPDX-License-Identifier: MIT									 *
 *  Copyright (C) 2006-2024 by Helge Kortendieck  alan@kasmithee.de  *
 *                                                                   *
 *  This file is part of qrsyncmgr.                                  *
 *********************************************************************/
#include "rsync.h"

rsync::rsync( QObject *parent ) : QObject( parent)
{
	bErr = false;
	bRStop = false;
	bStarted = false;
}

rsync::~rsync()
{
}

void rsync::rsyChkProgs()					// check for rsync and x11-ssh-askpass
{
	iErr = 0;
	QFileInfo qfi( "/usr/bin/rsync" );			// check for rsync
	if( qfi.exists() == true ) {
		sMsg = tr( "rsync found - OK !\n" );
		iErr += 1;
	} else {
		sMsg = tr("rsync not found !\n"
				  "rsync is either not installed or the program is not in your path.\n" );
	}
//cout << "rsyChkProgrs reached" << QString::number(iD).toStdString() << endl;                                               // *****  debug
	emit sigUpdOutput(sMsg, 1);				// update of widget in qrsyncmgr widget

/*
	sAskPassPath = "";
	slArgs.clear();
	slArgs << "/usr/bin/x11-ssh-askpass" << "/usr/bin/ssh-askpass" << "/usr/lib/ssh/x11-ssh-askpass" << "/usr/lib/ssh/ssh-askpass" << "/usr/lib/openssh/x11-ssh-askpass" << "/usr/lib/openssh/ask-pass";
	for (iD = 0; iD < slArgs.count(); iD++) { 
		QFileInfo qfi(slArgs.at(iD));		// check for askpass
		if (qfi.exists() == true) {
			sD = slArgs.at(iD);
			iD2 = sD.lastIndexOf("/");
//			sAskPass = sD.right(sD.length() - 1 - iD2);
			if (iD2 > 1) {							// only add environment to qprocess if askpass not in /usr/bin (this should always be part of users PATH)
				sAskPassPath = sD.left(iD2 + 1);	// including closing "/"
			}
			sMsg = tr("ssh-askpass found - OK !\n");
			iErr += 2;
			break;
		}
	}
	if (iErr < 2) {
		sMsg = tr("ssh-askpass not found !\n"
				  "rsync cannot ask you for the login password of a remote machine.\n"
				  "rsync via ssh will only work if the remote machine's ssh login key is stored on this machine.\n");
	}
	emit sigUpdOutput(sMsg, 1);				// update of widget in qrsyncmgr widget
*/

	emit sigChkEnd(iErr);
}


void rsync::rsyPrep(const QString& sD)    // prepare rsync, QString can have many lines (rsync definitions)
{
  sRsys = sD;
  slRsys.clear();
  slRsys = sRsys.split(QRegularExpression("\n"));
  iRsys = slRsys.count();
  if (iRsys > 1 and slRsys.last().isEmpty() == true) {
    slRsys.removeLast();
    iRsys = iRsys - 1;
  }
  if (iRsys > 0) {
    iRsy = 1;
    sLine = slRsys.at(iRsy - 1).simplified();
    procPrep();
  }
}

void rsync::procPrep()						// prepare rsync run and digest the rsync command line
{
//  sLine = sD.simplified();
  if (sLine.left(6) == "rsync ") {
    sLine = sLine.remove(0, 6).trimmed();
  }
  if (sLine.length() > 0) {
    iArgs = 0;									// argument count, need at least 3: option, src and dest
    slArgs.clear();
    emit sigUpdOutput(tr("Adding rsync arguments..."), 1);	// sent rsync message to output widget
    while (sLine.length() > 0) {
      iD = sLine.indexOf(" ");
      if (iD > 0) {
        sD = sLine.left(iD);
        slArgs.append(sD);
        iArgs = iArgs + 1;
        sLine.remove(0 , iD + 1);
        sLine = sLine.trimmed();
      } else {									// assuming last command
        sD = sLine.left(iD);
        slArgs.append(sD);
        iArgs = iArgs + 1;
        sLine.clear();
      }
      emit sigUpdOutput("     " + sD, 1);		// sent rsync argument to output widget
    }
  } else {										// rsync command line is empty
    sErrT = tr("Error - no rsync command");
    sErr = tr("The command line for rsync is empty!\n \nAborting!   (hit OK)");
    emit sigErr(sErrT, sErr);					// no rsync command, show error pop up
    sMsg = tr("Error - no rsync command - aborted !");
    emit sigUpdOutput(sMsg, 1);					// sent rsync start time to output widget
    return;
  }

  if (iArgs < 3) {
    sErrT = tr("Error - less than 3 arguments given");
    sErr = tr("The command line for rsync holds less than the minimum of 3 arguments!\n \nAborting!   (hit OK)");
    emit sigErr(sErrT, sErr);					// failed to setup rsync
    sMsg = tr("Error - less than 3 arguments given for rsync command - aborted !");
    emit sigUpdOutput(sMsg, 1);					// sent rsync start time to output widget
  } else {
    procStart();
  }
}


void rsync::procStart()						// run rsync process
{
	iErr = 0;									// internal error code, set in read stdout and stderr
	iErr32 = 0;
	iErr64 = 0;
	bErr = false;
	bStarted = false;
	bRStop = false;
	sLastRun = "";
	iArgs = 0;

	QTime t;
	sMsg = tr("rsync %1/%2 started at: ").arg( QString::number(iRsy) ).arg(QString::number(iRsys)) + t.currentTime().toString("hh:mm:ss") + "\n ";
	emit sigUpdOutput(sMsg, 1);					// sent rsync start time to output widget

	elap.currentTime();							// set time elap to current time to get rsync duration

	rsy = new QProcess(this);
	connect(rsy, SIGNAL(readyReadStandardOutput()), this, SLOT(procReadStdOut()));
	connect(rsy, SIGNAL(readyReadStandardError()), this, SLOT(procReadStdErr()));
	connect(rsy, SIGNAL(finished(int, QProcess::ExitStatus)), this, SLOT(procExit(int, QProcess::ExitStatus)));
	QProcessEnvironment qenv = QProcessEnvironment::systemEnvironment();
	qenv.insert("PATH", qenv.value("PATH") + sAskPassPath);
	rsy->setProcessEnvironment(qenv);
	rsy->start("rsync", slArgs);
}

void rsync::procReadStdOut()				// read Stdout of rsync process
{
  aOutput = rsy->readAllStandardOutput();					// read byte array
  if ( !aOutput.isEmpty() ) {
    sMsg = "stdout >>>  " + QString( aOutput ).trimmed();	// catch output line
    emit sigUpdOutput( sMsg, 0 );							// update of widget in rsynkmgrwidget
    if( !bStarted ) {
      if( sMsg.indexOf( "file list" , 0 ) > -1 ) {			// rsync has started, disable error/password handling
        bStarted = true;
      }
    }
  }
}

void rsync::procReadStdErr()				// read Stdout of rsync process
{
  aOutput = rsy->readAllStandardError();					// read byte array
  if (!aOutput.isEmpty()) {
    sMsg = "stderr >>>  " + QString(aOutput).trimmed();	// catch output line
    emit sigUpdOutput(sMsg, 0);							// update of widget in qrsyncmgr widget
    if (sMsg.indexOf("(code 23)" , 0) > -1) {			// rsync error code 23, some files not transfered
      iErr = 1;
    }
    if (bStarted == false) {
      if (sMsg.left(13) != "(ssh-askpass:" and sMsg.indexOf("ssh-askpass" , 0) > -1) {	// ssh password failure, not a gtk ssh-askpass failure
        iErr = 2;
      } else if (sMsg.indexOf("Host key verification failed" , 0) > -1) {	// ssh password failure
        iErr = 4;
      } else if (sMsg.indexOf("(code 127)" , 0) > -1) {						// rsync not found on remote machine
        iErr = 8;
      } else if (sMsg.indexOf("Permission denied" , 0) > -1) {				// ssh password wrong or cancel pressed - asked to stop or continue
        emit sigUpdOutput(tr("In case you hit 'Cancel': You must cancel 3 times, this is not within qrsyncmgr's control!"), 1); // update output in qrsyncmgr
        iErr = 32;
      } else if (sMsg.indexOf("password" , 0) > -1) {	// assume getting error on entering password - retry
        iErr = 16;
      } else if (sMsg.indexOf("file list" , 0) > -1) {	// rsync has started, disable error/password handling
        bStarted = true;
      }

      if (iErr > 0) {
        bErr = true;
      }
    } else {									// rsync has started
      if (sMsg.indexOf("rsync:" , 0) > -1 and  sMsg.indexOf("failed" , 0) > -1 and sMsg.indexOf("(1)" , 0) > -1) {		// dir permission denied
        iErr32 = 64;
      }
      if (sMsg.indexOf("rsync:" , 0) > -1 and  sMsg.indexOf("failed:" , 0) > -1 and sMsg.indexOf("(13)" , 0) > -1) {	// dir permission denied
        iErr64 = 128;
      }
    }
  }
/*  if( iErr = 32 ) {
    procPwErr();
  }
*/
}

void rsync::procPwErr()						// wrong ssh password or user canceled password input
{
	rsy->terminate();
	QTimer::singleShot(2000, rsy, SLOT( kill()));	// closing delay to allow to exit gracefully
}

void rsync::procExit(int iD, QProcess::ExitStatus iD2)	// rsync process has exited
{
  disconnect(rsy, SIGNAL(readyReadStandardOutput()), this, SLOT(procReadStdOut()));
  disconnect(rsy, SIGNAL(readyReadStandardError()), this, SLOT(procReadStdErr()));
  disconnect(rsy, SIGNAL(finished( int, QProcess::ExitStatus)), this, SLOT(procExit(int, QProcess::ExitStatus)));
  if (rsy->exitStatus() == QProcess::NormalExit and !bErr and !bRStop) {    // true if normal exit,  no error in Output found and not stopped by user
    if (iRsy < iRsys) {                        // another rsync to run, defined in this definition
      iRsy = iRsy + 1;
      sLine = slRsys.at(iRsy - 1).simplified();
      procPrep();
    } else {                                    // no more rsyncs to run in this definition
      QTime t;
      emit sigUpdOutput("", 0);               // empty line
      sMsg = tr("rsync ended at: %1 . Appears to be successfully!").arg(t.currentTime().toString("h:mm:ss"));
      iD = elap.secsTo(QTime::currentTime()) * -1;
      iHr = iD / 1000 / 60 / 60;                  // get minutes
      iD = iD - (iHr * 60 * 60 * 1000);
      iMin = iD / 1000 / 60;
      iD = iD - (iMin * 60 * 1000);
      iSec = iD / 1000;
      sMsg = sMsg + tr(" Duration: %1.%2:%3").arg(QString::number(iHr)).arg(QString::number(iMin)).arg(QString::number(iSec));
      QDateTime dt;
      sLastRun = dt.currentDateTime().toString("dd. MMM yyyy, hh:mm");
      emit sigUpdOutput(sMsg, 1);                 // update output widget in qrsyncmgr
      sErr = "";
      sMsg = "";
      if (iErr == 1) {                             // some files could not be transferred warn ! warning and error
        sMsg = tr("Warning ! Some files could not be transferred ! rsync error (23).\n"
                  "Check the output widget for more information.\n");
        sErr = sMsg;
        sMsg = sMsg + "\n";
      }
      if (iErr32 == 64) {                          // warning only, do not add to error pop up
        sMsg = sMsg + tr("Warning ! rsync failed to set times in one or more directories! rsync error (1).\n"
                         "This is probably none fatal and can be caused by mounting a partition with the 'noatime' option.\n"
                         "Files in this directory are likely still synced/saved.\n"
                         "See this output widget for more information.\n");
      }
      if (iErr64 == 128) {                          // error
        sErr = sErr + tr("Error ! rsync mkstemp failed on one or more files! rsync error (13).\n"
                         "This indicates a file permission problem !\n"
                         "These files were NOT synced/saved !\n"
                         "See the output widget for the affected files.\n");
      }
      if (iErr == 1 or iErr32 == 64) {             // warning, update output widget in qrsyncmgr
        emit sigUpdOutput(" ", 0);                    // add empty line
        emit sigUpdOutput(sMsg, 1);                   // warnings
      }
      if (iErr == 0 or iErr == 1 or iErr64 == 128) {    // ended, if error, show error pop up widget
        emit sigOk(sLastRun, iErr, sErr);
      }
    }
  } else if (bRStop) {
    bRStop = false;
    sMsg = tr("rsync stopped on user request !");
    emit sigUpdOutput(sMsg, 1);                   // update output widget in qrsyncmgr
    emit sigStopped();
  } else {
    switch (iErr) {
      case 2:                                   // ssh-askpass not found
        sErr = tr("rsync requires ssh-askpass which is currently not installed.\n"
                  "Either the stored login password key for the remote machine is (no longer?) valid\n"
                  "or you need to install ssh-askpass-gnome (or ssh-askpass) to be able\n"
                  "to enter the password when starting the rsync.\n \n"
                  "Aborting!   (hit OK)");
        break;
      case 4:                                   // ssh login failure
        sErr = tr("Login via ssh failed.\n"
                  "Either the stored login password key for the remote machine is (no longer?) valid\n"
                  "or the password you entered was wrong.\n \n"
                  "Aborting!");
        break;
      case 8:                                   // remote command not found - rsync not installed on remote PC
        sErr = tr("Remote command not found.\n"
                  "This error indicates that rsync is not installed on the remote machine.\n"
                  "Check whether it is installed there too and in the path of the user you logged on as.\n \n"
                  "Aborting!");
        break;
      case 16:                                   // public key or password found, wrong ssh password for log into remote PC
        sErr = tr("There seems to be a problem with the public key or password!\n"
                  "Did you provide the correct ssh password?\n"
                  "Hopefully the output window can tell you more...\n \n"
                  "Aborting!");
        break;
      case 32:                                   // ssh password wrong or canceled by user
        sErr = tr("There seems to be a problem with the public key or password!\n"
                  "Either the ssh password was wrong or entering the password was canceled!\n"
                  "In case you entered the wrong password, please re-run this rsync.\n \n"
                  "Aborting!");
        break;
      default:
        sErr = tr("Sorry, it appears, that an unknown error occurred while running rsync!\n"
                  "Hopefully the output window can tell you more...\n \n"
                  "Aborting!");
    }
    procErr();									// send rsync error messages
  }
}

void rsync::procStop()						// stop rsync process on user request
{
	bRStop = true;
	rsy->terminate();
	QTimer::singleShot(1000, rsy, SLOT( kill()));
}

void rsync::procErr()						// error, during rsync run, send error message
{
	sErrT = tr("Error - while rsyncing");
	emit sigErr(sErrT, sErr);					// sent rsync error messages to show
}


//#include "rsync.moc"
