/*********************************************************************
 *  SPDX-License-Identifier: MIT									 *
 *  Copyright (C) 2006-2026 by Helge Kortendieck  alan@kasmithee.de  *
 *                                                                   *
 *  This file is part of qrsyncmgr.                                  *
 *********************************************************************/
#include "qrm.h"

Qrmgr::Qrmgr(QMainWindow *parent)
        : QMainWindow(parent)
{
	bQuit = false;          	// flag whether we want to exit

	bDefChgd = false;
	bRwdNewEntry = false;

	iRsync = 0;             	// select first rsync after boot
	iSyncing = 0;				// rsync status undefined
	iWShown = 0;
  
	ui.setupUi(this);
	connect(ui.pB_DefDir, SIGNAL(clicked()), this, SLOT(setDir()));        // open file dialog to set directory with rsync definitions
	connect(ui.lW_Rsyncs, SIGNAL(itemClicked(QListWidgetItem*)), this, SLOT(clickedItem(QListWidgetItem*))); // rsync def in left list widget clicked
	connect(ui.pB_DefClr, SIGNAL(clicked()), this, SLOT(clickedDefClr())); // clicked on clear form from rsync definitions
	connect(ui.pB_DefAdd, SIGNAL(clicked()), this, SLOT(clickedDefAdd())); // clicked on add new rsync definition
	connect(ui.pB_DefUpd, SIGNAL(clicked()), this, SLOT(clickedDefUpd())); // clicked on update current rsync definition
	connect(ui.pB_DefDel, SIGNAL(clicked()), this, SLOT(clickedDefDel())); // clicked on delete current rsync definition

	connect(ui.lE_Name, SIGNAL(textChanged(const QString&)), this, SLOT(defChgd()));     // def was changed
	connect(ui.tE_Desc, SIGNAL(textChanged()), this, SLOT(defChgd()));                   // text changed in rsync def command(s)
//	connect(ui.lE_SrcMp, SIGNAL(textChanged(const QString&)), this, SLOT(defChgd()));    // def was changed
//	connect(ui.lE_DestMp, SIGNAL(textChanged(const QString&)), this, SLOT(defChgd()));   // def was changed
	connect(ui.tE_Args, SIGNAL(textChanged()), this, SLOT(defChgd()));                   // text changed in rsync def command(s)
	ui.pB_DefDir->setIcon(QIcon("/usr/share/qrsyncmgr/icons/scalable/open.svg"));
	ui.l_Desc_Warn->setPixmap(QIcon("/usr/share/qrsyncmgr/icons/scalable/warning.svg").pixmap(32,32));
	ui.l_Args_Info->setPixmap(QIcon("/usr/share/qrsyncmgr/icons/scalable/info.svg").pixmap(32,32));

	rwd = new rwdefs();                       // create rwdefs qobject
	connect(rwd, SIGNAL(sigUpdOutput(const QString&, int)), this, SLOT(addOutput(const QString&, int)));  // update output widget w rwdefs output
	connect(rwd, SIGNAL(sigDirChkd(const QString&, const QString&, int)), this, SLOT(initChkRsyncDirRet(const QString&, const QString&, int))); // return from checking dirs
	connect(rwd, SIGNAL(sigErr(const QString&, const QString&)), this, SLOT(rwdErr(const QString&, const QString&)));	// error from rwdefs qobject
	connect(rwd, SIGNAL(sigRsyncsRead(const QStringList&)), this, SLOT(rwdUpdList(const QStringList&)));  // rsync names read, update widget with list
	connect(rwd, SIGNAL(sigUpdDetails(const Rsync&)), this, SLOT(rwdUpdDetails(const Rsync&)));   // get and show details of selected rsync
	connect(this, SIGNAL(sigRwdAdd(const Rsync&)), rwd, SLOT(rsyncAdd(const Rsync&)));            // add rsync definition
	connect(this, SIGNAL(sigRwdChkDir(const QString&)), rwd, SLOT(dirCheck(const QString&)));     // check directory with rsync definitions
	connect(this, SIGNAL(sigRwdUpd(const Rsync&)), rwd, SLOT(rsyncUpd(const Rsync&)));            // update rsync definition
	connect(this, SIGNAL(sigRwdDel(const QString&)), rwd, SLOT(rsyncDel(const QString&)));        // delete rsync definition
	connect(this, SIGNAL(sigRwdGetDetails(const QString&)), rwd, SLOT(rsyncGet(const QString&))); // get rsync details (fields)
	connect(this, SIGNAL(sigRwdGetRsyncs()), rwd, SLOT(readRsyncs()));  // read Name(s) from rsyncs, all defined rsync labels

	rsy = new rsync();
	connect(rsy, SIGNAL(sigUpdOutput(const QString&, int)), this, SLOT(addOutput(const QString&, int)));	// update output widget
	connect(rsy, SIGNAL(sigChkEnd(int)), this, SLOT(initChkProgsRet(int)));									// return from checking for rsync n ask-pass
	connect(rsy, SIGNAL(sigErr(const QString&, const QString&)), this, SLOT(rsyncErr(const QString&, const QString&)));				// error from rsync qobject
	connect(rsy, SIGNAL(sigOk(const QString&, int, const QString&)), this, SLOT(rsyncEnd(const QString&, int, const QString&)));	// rsync end successful
	connect(rsy, SIGNAL(sigStopped()), this, SLOT(rsyncStopped()));							// rsync stopped
	connect(this, SIGNAL(sigRsyChkProgs()), rsy, SLOT(rsyChkProgs()));						// check for installed rsync and ask-pass
	connect(this, SIGNAL(sigRsyStart(const QString&)), rsy, SLOT(rsyPrep(const QString&)));	// start rsync(s)
	connect(this, SIGNAL(sigRsyStop()), rsy, SLOT( procStop()));							// signal to stop rsync

	toolBar = addToolBar(tr("Show toolbar"));
	settingsRead();

	actionsCreate();
	updMnTlBr();
	statusBarCreate();

	initChkRsyncDir();
}

Qrmgr::~Qrmgr()
{
}


//  *****************************************************************************
//  **********  settings read/write                                   ***********
//  *****************************************************************************
void Qrmgr::settingsRead()					// read settings (for client mode)
{
	QSettings settings("qrsyncmgr", "qrsyncmgr");

	settings.beginGroup( "mainwindow" );
		QPoint pos = settings.value("pos", QPoint(0, 0)).toPoint();
		QSize size = settings.value("size", QSize(800, 600)).toSize();
		ui.splitterH->restoreState(settings.value("splitterHori").toByteArray());
		ui.splitterV->restoreState(settings.value("splitterVert").toByteArray());
		sDir = settings.value("dir", QDir::homePath()).toString();  // directory with rsync definition
	settings.endGroup();

	resize(size);
	move(pos);
	ui.l_DefDir->setText(sDir);				// set dir with definitions
}

void Qrmgr::settingsWrite()					// write settings
{
	QSettings settings("qrsyncmgr", "qrsyncmgr");

	settings.beginGroup("mainwindow");
		settings.setValue("pos", pos());
		settings.setValue("size", size());
		settings.setValue("splitterHori", ui.splitterH->saveState());
		settings.setValue("splitterVert", ui.splitterV->saveState());
		settings.setValue("dir", sDir);
	settings.endGroup();

	settings.sync();
}

//  *****************************************************************************
//  **********  actions, menu and toolbar                             ***********
//  *****************************************************************************
void Qrmgr::actionsCreate()
{
	startAct = new QAction(QIcon("/usr/share/qrsyncmgr/icons/scalable/start.svg"), tr("Start selected rs&ync"), this);
	startAct->setShortcut (tr("Ctrl+Y"));
	startAct->setStatusTip(tr("Start selected rsync"));
	connect(startAct, SIGNAL(triggered()), this, SLOT(clickedStart()));

	disabledAct = new QAction(QIcon("/usr/share/qrsyncmgr/icons/scalable/disabled.svg"), tr("Starting a rsync not possible"), this);
	disabledAct->setStatusTip(tr("Starting a rsync not possible - rsync not installed or a rsync is currently running."));

	stopAct = new QAction(QIcon("/usr/share/qrsyncmgr/icons/scalable/stop.svg"), tr("St&op current rsync"), this);
	stopAct->setShortcut(tr("Ctrl+O"));
	stopAct->setStatusTip(tr("Stop current rsync"));
	connect(stopAct, SIGNAL(triggered()), this, SLOT(clickedStop()));

	exitAct = new QAction(QIcon("/usr/share/qrsyncmgr/icons/scalable/exit.svg"), tr("&Quit the program"), this);
	exitAct->setShortcut(tr("Ctrl+Q"));
	exitAct->setStatusTip(tr("Quit the application"));
	connect(exitAct, SIGNAL(triggered()), this, SLOT(clickedQuit()));

	helpAct = new QAction(QIcon("/usr/share/qrsyncmgr/icons/scalable/help.svg"), tr("&Help"), this);
	helpAct->setShortcut(tr("Ctrl+H"));
	helpAct->setStatusTip(tr("Show the application's online help"));
	connect(helpAct, SIGNAL(triggered()), this, SLOT(help()));

	aboutAct = new QAction(QIcon("/usr/share/qrsyncmgr/icons/scalable/about.svg"), tr("&About qrsyncmgr"), this);
	aboutAct->setShortcut(tr("Ctrl+A"));
	aboutAct->setStatusTip(tr("Show the application's about information"));
	connect(aboutAct, SIGNAL(triggered()), this, SLOT(about()));
}

void Qrmgr::menuCreate()
{
	menuBar()->clear();
	fileMenu = menuBar()->addMenu(tr("&File"));
	fileMenu->addAction(exitAct);
	menuBar()->addSeparator();
	rsyncMenu = menuBar()->addMenu(tr("&Rsync"));
	if (iSyncing == 0 or iSyncing == 1) {			// rsync undefined or not found/unavailable
		rsyncMenu->addAction(disabledAct);
	} else if (iSyncing == 2) {						// not syncing
		rsyncMenu->addAction(startAct);
	} else if (iSyncing == 3) {						// rsync running
		rsyncMenu->addAction(stopAct);
	}
	menuBar()->addSeparator();
	helpMenu = menuBar()->addMenu(tr("&Help"));
	helpMenu->addAction(helpAct);
	helpMenu->addAction(aboutAct);
}

void Qrmgr::toolBarCreate()
{
	toolBar->clear();
	toolBar->addAction(exitAct);
	toolBar->addSeparator();
	if (iSyncing == 0 or iSyncing == 1) {		// syncing not possible, undefined or rsync not found
		toolBar->addAction(disabledAct);
	} else if (iSyncing == 2) {				// not syncing
		toolBar->addAction(startAct);
	} else if (iSyncing == 3) {				// syncing in progress
		toolBar->addAction(stopAct);
	}
}

void Qrmgr::statusBarCreate()
{
	statusBarUpd(tr("Ready"));
}

void Qrmgr::updMnTlBr()						// update menu and toolbar
{
	menuCreate();
	toolBarCreate();
}

void Qrmgr::statusBarUpd(const QString& sD)
{
	statusBar()->showMessage(sD);
}


//  *****************************************************************************
//  **********  init - initial checks at start up - rsyncs, programs  ***********
//  *****************************************************************************
void Qrmgr::initChkRsyncDir()				// check dir with rsyncs
{
	emit sigRwdChkDir(sDir);					// check dir with rsync definitions and return with list of names if any
}

void Qrmgr::initChkRsyncDirRet(const QString& sD, const QString& sD2, int iD)	// return from rsync directory check
{
	iErrDirChk = iD;                    		// error status while checking dir w rsync defs: exists 0/1 + readable 0/2 + writeable 0/4 + has defs 0/8 = 0,1,3,7,11,15
	if (iErrDirChk < 15) {						// have error, show problem
		QMessageBox::warning(this, sD, sD2);
	}

//cout << "appScreen & numScreens: " << QString::number( iD ).toStdString() << " : " << QString::number( iD2 ).toStdString() << endl;                                               // *****  debug

	if (iErrDirChk < 4 or iErrDirChk == 11) {	// disable definition buttons
		butDef(0);
	} else {									// enable rsync definition buttons
		butDef(1);
	}

	if (iErrDirChk < 11) {
		iSyncing = 0;							// rsync undefined, cannot run rsync
	} else {
		iSyncing = 2;							// rsync not running, enable rsync start
	}
	updMnTlBr();
//cout << "iErrDirChk = " << QString::number(iErrDirChk).toStdString() << endl;                                               // *****  debug
	initChkProgs();								// now check for required programs
}

void Qrmgr::initChkProgs()					// check for rsync
{
	emit sigRsyChkProgs();
//	QTimer::singleShot( 500, rsy, SLOT(rsyChk()) );	// required to show program widget if rsync not installed - error comes too fast otherwise !
}

void Qrmgr::initChkProgsRet(int iD)			// check for rsync return, 0=not found 1=found
{
//cout << "initChkProgsRet = " << QString::number(iD).toStdString() << endl;                                               // *****  debug
	if (iD != 1) {
		QMessageBox::warning(this, tr("qrsyncmgr fatal error!"), tr("Could not find rsync (/usr/bin/rsync)!\n"
																		"rsync might not be installed. Aborting..."));
		quitDo();
		return;
	}
	initFin();
}

void Qrmgr::initFin()						// finalize init
{
	iOpCount = 0;								// counter for output lines
}

void Qrmgr::about()
{
	wCtrl(1);
}

void Qrmgr::help()
{
	wCtrl(2);
}

void Qrmgr::quitPrep()                      // quit clicked clean up
{
	settingsWrite();
	clickedStop();								// stop any running rsync and/or unmount mounted partitions
}

void Qrmgr::quitDo()						// finally quit the program
{
	exit(0);
}

void Qrmgr::getCurPos()						// get cursor position to show a message close by
{
	qPos = QCursor::pos();
	iX = qPos.x() - 150;
	if (iX < 0) {
		iX = 0;
	}
	iY = qPos.y() - 150;
	if (iY < 0) {
		iY = 0;
	}
}

// ********************************************************************************************************************
// **  clicked button collection         **
// ****************************************
void Qrmgr::clickedStart()                  // start button was clicked
{
	iSyncing = 3;
	updMnTlBr();
	butDef(0);                                  // disable definitions
	butDef(33);
	ui.tE_Output->clear();
	iOpCount = 0;                                 // counter for output lines
	sRsync = ui.lW_Rsyncs->currentItem()->text();
	rsyncStart();
//	mntStart();
}

void Qrmgr::clickedStop()                   // stop button was clicked
{
	if (iSyncing == 3) {
		getCurPos();
		QMessageBox msgBox;
		msgBox.setWindowTitle(tr("Stop running rsync ?"));
		msgBox.setIcon(QMessageBox::Question);
		msgBox.setText(tr("\nThere seems to be a rsync still running.\n"
							"Sure to stop ?\n" ) );
		msgBox.addButton(tr("Yes, &Stop"), QMessageBox::AcceptRole);
		msgBox.addButton(tr("No, &Cancel"), QMessageBox::RejectRole);
		msgBox.move(iD, iD2);
		if (msgBox.exec() == 0) {
			emit sigRsyStop();
		} else {								// No, don't stop
			bQuit = false;						// do not exit
		}
	} else if (bQuit == true) {				// clickedStop was called from exit routine
		quitDo();								// exit
	}
}

void Qrmgr::clickedQuit()					// Quit button was clicked
{
	getCurPos();
	QMessageBox msgBox;
	msgBox.setWindowTitle(tr("Quit ?"));
	msgBox.setIcon(QMessageBox::Question);
	msgBox.setText(tr( "\nSure to quit ?\n"));
	msgBox.addButton(tr("Yes, &Quit"), QMessageBox::AcceptRole);
	msgBox.addButton(tr("No, &Cancel"), QMessageBox::RejectRole);
	msgBox.move(iX, iY);
	iRet = msgBox.exec();						// 2=accept, 3=reject
	if (iRet == 2) {
		bQuit = true;
		quitPrep();
	}
}

void Qrmgr::clickedItem(QListWidgetItem* qlwiD) // qlistwidget item was clicked
{
	if (bDefChgd == true) {
		QMessageBox msgBox;
			msgBox.setWindowTitle(tr("qrsyncmgr - save changes ?"));
			msgBox.setIcon(QMessageBox::Question);
			msgBox.setText(tr("The current rsync definition has pending changes.\n \n"
							  "Do you want to save the changes before switching to the selected definition?\n"));
			msgBox.addButton(tr("Yes, abort"), QMessageBox::AcceptRole);
			msgBox.addButton(tr("No, continue"), QMessageBox::RejectRole);
//			msgBox.move(iX, iY);
		if(msgBox.exec() == QMessageBox::RejectRole) {
			rwdSetCurIt(ui.lE_Name->text());
			return;
		}
	}
	rwdClearForm();
	rwdGetDetails();
}

void Qrmgr::clickedDefClr()                 // clicked clear rsync definition form
{
  rwdClearForm();
  butDef(2);
}

void Qrmgr::clickedDefAdd()                 // clicked add rsync definition
{
  sD = ui.lE_Name->text();
  if (sD.indexOf("/") > -1) {
    QMessageBox::information(this, tr("Invalid name !"), 
                                    tr("The file name cannot contain the character '/'.\n"
                                        "Please change the name.\n"
                                        "You can use this character in the description field."));
    return;
  }
  iAddDelUpd = 0;
  rwdDefAddDelUpd();
}

void Qrmgr::clickedDefUpd()					// clicked upd rsync definition
{
	iAddDelUpd = 2;
	rwdDefAddDelUpd();
}

void Qrmgr::clickedDefDel()                 // clicked del rsync definition
{
  if (!QMessageBox::question(0, tr("qrsyncmgr - sure to delete rsync?"),
								tr("Are your sure to delete this rsync (definition)?\n \n"
                                   "Hit 'OK' to delete or 'Cancel' to abort."), 
								QMessageBox::Yes | QMessageBox::No, QMessageBox::NoButton)) {
    iAddDelUpd = 1;
    iRsync = ui.lW_Rsyncs->currentRow();
    rwdDefAddDelUpd();
  }
}

void Qrmgr::setDir()					// set directroy with rsync definitions, file open dialog button was clicked
{
  sD = QFileDialog::getExistingDirectory(this, tr("Set directory with rsync definitions"),
                                               sDir,
                                               QFileDialog::DontResolveSymlinks);
  if (sD != "") {
    sDir = sD;
    ui.l_DefDir->setText(sDir);				// set dir with definitions
    rwdClearForm();							// clear details form
    emit sigRwdChkDir(sDir);				// check dir with rsync definitions and return with list of names if any
  }
}

// ********************************************************************************************************************
// **  popup widget handling        **
// ***********************************
void Qrmgr::wCtrl(int iWToShow) 		// show popup widget, if no other is shown: 0=none, 1=about-abouthelp, 2=help-abouthelp, 3=error,no sys tray, 9=settings dialog
{
	if (iWShown == 0) {						// only proceed, if no other widget is shown
		switch (iWToShow) {
			case 1:							// show about widget
				ahwCreate();
				iWShown = 1;
				emit sigAhwShow(iWShown);
				break;
			case 2:							// show help widget
				ahwCreate();
				iWShown = 2;
				emit sigAhwShow(iWShown);
				break;
		}
	}
}

// ********************************************************************************************************************
// **  about/help/error widget handling  **
// ****************************************
void Qrmgr::ahwCreate()						// create about/help/error dialog widget
{
	if (iWShown == 0) {
		ahw = new AboutHelp();
		connect(this, SIGNAL(sigAhwShow(int)), ahw, SLOT(dlgShow(int)));
		connect(ahw, SIGNAL(sigDelAhw()), this, SLOT(ahwDel()));
	}
}

void Qrmgr::ahwDel()						// ahw widget was hidden
{
	ahw->deleteLater();
	iWShown = 0;
}


void Qrmgr::addOutput(const QString& sD, int iD)	// add output to widget, iD=1 = qrsyncmgr message
{
	sMsg = sD;
	iQRM = iD;
	if (iOpCount > 750) {
		ui.tE_Output->clear();
		iOpCount = 0;
	}
	if (iQRM == 1) {
		sMsg = tr(">>>   qrsyncmgr:   %1").arg(sMsg);
	}
	ui.tE_Output->append(sMsg);  				// append output to widget
	iOpCount = iOpCount + 1;
//cout << "addOutput = " << endl;                                               // *****  debug
}

void Qrmgr::defChgd()						// rsync definition has changed
{
  bDefChgd = true;
  butCtrl();
}

void Qrmgr::butCtrl()						// set buttons acc. rsync definition filed updates
{
  sD = ui.lE_Name->text();
  if (sD != "") {
    bD = false;
    for (int i = 0; i < ui.lW_Rsyncs->count(); ++i) {
      if (ui.lW_Rsyncs->item(i)->text() == sD) {
        bD = true;								// definition is currently defined, only update, not add new
      }
    }
    if (bD == true) {							// def exists
      butDef(2);									// enable clear form button
      butDef(5);									// disable add def button
      butDef(8);									// enable upd button
      butDef(16);									// enable delete button
    } else {									// def does not exist
      butDef(2);									// enable clear form button
      butDef(4);									// enable add def button
      butDef(9);									// disable upd button
      butDef(17);									// enable delete button
    }
  }

	if (ui.lE_Name->text() == "" or ui.tE_Args->toPlainText() == "") {
		butDef(2);								// enable clear form button
		butDef(5);								// disable add def button
		butDef(9);								// disable upd button
		iSyncing = 0;
		updMnTlBr();							// disable rsync
		sD = tr("rsync definition incomplete, name and/or rsync commands field(s) are empty !");
		ui.pB_DefAdd->setToolTip(sD);
		ui.pB_DefUpd->setToolTip(sD);
	} else {
		iSyncing = 2;
		updMnTlBr();							// enable rsync
		ui.pB_DefAdd->setToolTip(tr("Add this definition to the list, name must be unique"));
		ui.pB_DefUpd->setToolTip(tr("Update this definition"));
	}
}

void Qrmgr::butDef(int iRwdButs)			// disable/enable rsync def buttons
{
	switch (iRwdButs) {
		case 0:									// disable buttons
			ui.f_Details->setEnabled(false);
			break;
		case 1:									// enable buttons
			ui.f_Details->setEnabled(true);
			break;
		case 2:									// enable clear form button
			ui.pB_DefClr->setEnabled(true);
			break;
		case 3:									// disable clear form button
			ui.pB_DefClr->setEnabled(false);
			break;
		case 4:									// enable add def button
			ui.pB_DefAdd->setEnabled(true);
			break;
		case 5:									// disable add def button
			ui.pB_DefAdd->setEnabled(false);
			break;
		case 8:									// enable upd def button
			ui.pB_DefUpd->setEnabled(true);
			break;
		case 9:									// disable upd def button
			ui.pB_DefUpd->setEnabled(false);
			break;
		case 16:								// enable del def button
			ui.pB_DefDel->setEnabled(true);
			break;
		case 17:								// disable del def button
			ui.pB_DefDel->setEnabled(false);
			break;
		case 32:								// enable set directory with rsyncs button
			ui.pB_DefDir->setEnabled(true);
			break;
		case 33:								// disable set directory with rsyncs button
			ui.pB_DefDir->setEnabled(false);
			break;
	}
}


// ***********************************************************
// **  functions to read/write rsync definitions            **
// ***********************************************************
void Qrmgr::rwdErr(const QString& sD, const QString& sD2)	// show error from rwdefs qobject
{
	QMessageBox::information(this, sD, sD2);
	butDef(1);									// enable buttons again
	butDef(32);									// enable select directory with rsync definitions
}

void Qrmgr::rwdGetDetails()				// get rsync details (fields) also called by signal lB_Rsyncs->currentChanged()
{
//QMessageBox::information(this, "debug rsyncGetDetails()", ui.lW_Rsyncs->currentItem()->text() );                  // **********  debug only  *****************
	emit sigRwdGetDetails(ui.lW_Rsyncs->currentItem()->text());
}

void Qrmgr::rwdClearForm()				// clear the rsync define widget to add a new form
{
	butDef(3);								// disable buttons
	butDef(5);								// disable buttons
	butDef(9);								// disable buttons
	butDef(17);								// disable buttons
  	iSyncing = 0;
	updMnTlBr();							// disable rsync
	ui.lE_Name->clear();
	ui.lE_Name->setFocus();
	ui.l_LastRun->clear();
	ui.tE_Desc->clear();
	ui.tE_Args->clear();
}

void Qrmgr::rwdDefAddDelUpd()			// add, del or update current rsync definition
{
	if (ui.lW_Rsyncs->count() > 0) {
		sRsync = ui.lW_Rsyncs->currentItem()->text();
	}
	rsyncD.name = ui.lE_Name->text();			// name of rsync
	rsyncD.lastRun = ui.l_LastRun->text();		// new, must be empty
	rsyncD.desc = ui.tE_Desc->toPlainText();	// rsync description
	rsyncD.cmd = ui.tE_Args->toPlainText();		// rsync command, excl. 'rsync'

	switch (iAddDelUpd) {
		case 0:								// add current rsync definition
			bRwdNewEntry = true;
			sRwdNewEntry = rsyncD.name;
			rsyncD.lastRun = "";			// new, must be empty
			emit sigRwdAdd(rsyncD);
			break;
		case 1:								// delete current rsync def
			iD = ui.lW_Rsyncs->count();
			if (iRsync == iD - 1) {			// last item: iRsync starts with index 0, count() starts with index 1
				sRsyncAD = ui.lW_Rsyncs->item(iRsync - 1)->text();	// deleting last item, select previous item after update
			} else {
				sRsyncAD = ui.lW_Rsyncs->item(iRsync + 1)->text();	// select next item after update
			}
			emit sigRwdDel(rsyncD.name);
			break;
		case 2:								// update current rsync def
			emit sigRwdUpd(rsyncD);
			break;
	}
}

void Qrmgr::rwdUpdDetails(const Rsync& rsyncD)	// update rsync details (fields)
{
//QMessageBox::information(this, "debug 1rwdUpdDetails", "reached" );                  // **********  debug only  *****************
	ui.lE_Name->setText(rsyncD.name);		// name of rsync
	ui.l_LastRun->setText(rsyncD.lastRun);	// last run date (string)
	ui.tE_Desc->setText(rsyncD.desc);		// rsync description
	ui.tE_Args->setText(rsyncD.cmd);			// rsync command, excl. 'rsync'
	butDef(2);								// enable clr button
	butDef(5);								// disable add button
	butDef(9);								// disable upd button
	butDef(16);								// enable del button
	bDefChgd = false;						// re-set change watchdog for rsync definition
}

void Qrmgr::rwdUpdList(const QStringList& slD)	// update widget with list of rsyncs
{
	slRsyncs = slD;
	ui.lW_Rsyncs->clear();
	if (slRsyncs.count() > 0) {						// found rsync definitions
		slRsyncs.sort();
		ui.lW_Rsyncs->insertItems(0, slRsyncs);
		if (bRwdNewEntry == true) {					// just added a new entry, select this
			sRsync = sRwdNewEntry;
			bRwdNewEntry = false;
		} else if (iRsync > slRsyncs.count() - 1) {	// if we have deleted the last entry, select the new last entry
			iRsync = slRsyncs.count() - 1;
			sRsync = slRsyncs.last();
		}

		if (iAddDelUpd == 1) {						// deleted a rsync
			sRsync = sRsyncAD;
		}
		rwdSetCurIt(sRsync);
		rwdGetDetails();							 // get details of selected rsync
	} else {
		iRsync = 0;
  }
}

void Qrmgr::rwdSetCurIt(const QString sD)	// set current item
{
	iRsync = slRsyncs.indexOf(sD, 0);
	if (iRsync == -1) {
		iRsync = 0;
	}
	ui.lW_Rsyncs->setCurrentItem(ui.lW_Rsyncs->item(iRsync));
}


// ***********************************************************
// **  start rsync - dirs should be already mounted         **
// ***********************************************************
void Qrmgr::rsyncStart()           // start rsync
{
	iSyncing = 3;
	emit sigRsyStart(ui.tE_Args->toPlainText());
}

void Qrmgr::rsyncErr(const QString& sErrT, const QString& sErr)  // show error from rsync qobject
{
	iSyncing = 2;									// not syncing
	QMessageBox::information(this, sErrT, sErr);
}

void Qrmgr::rsyncEnd(const QString& sD, int iD, const QString& sD2) // rsync has ended successfully: last run date, error, warning string
{
	sLastRun = sD;
	iErr = iD;
	iSyncing = 2;									// not syncing
	if (sLastRun.isEmpty() == false) {
		butDef(1);
		butDef(32);									// enable select directory with rsync definitions
		ui.l_LastRun->setText(sLastRun);
		clickedDefUpd();							// store last run date to rsync definition
		addOutput(tr( "Last run date saved.\n"), 1);	// append output to textEdit
		if( iErr == 1 ) {                           // rsync error code 23, not all files trasnferred
		QMessageBox::information(this, tr("Warning - rsync error 23"), 
									sD2 + tr("\nAnyhow, the 'Last run' date WAS stored!"));

		}
	} else {
		addOutput(tr( "Last run date could NOT be saved.\n"), 1);	// append output to textEdit
		QMessageBox::information(this, tr("Error - rsync"), 
									tr("Sorry, rsync seems to have ended successfully, \n"
										"but the 'Last Run' date could not be created?\n \n"
										"WARNING ! You better do not rely on this rsync !"));
	}

	if (bQuit == true) {
		addOutput(tr("Quitting in 2 seconds...\n"), 1); // append output to textEdit
		QTimer::singleShot(2000, this, SLOT(quitDo())); // allow user to read umnount finished in output widget
	} else {
		butDef(1);
		butDef(32);                               // enable select directory with rsync definitions
		iSyncing = 2;
		updMnTlBr();            	        		// enable rsync
	}
}

void Qrmgr::rsyncStopped()					// rsync has been stopped by user request		********************* still needed ??????????????
{
	iSyncing = 2;								// not syncing
	if (bQuit == true) {						// rsync stopped to exit
		quitDo();
	}
}


